/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: strlist.c,v 1.8 2003/10/24 23:26:52 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

#include <string.h>
#include <liblm.h>
#include "strlist.h"

int strlist_new(strlist_t * pstrlist, lm_string_t string)
{
    int i;
    int retval = 0;
    int len;
    int cur;
    tag_strlist_t *strlist = NULL;

    len = LM_STRING_LEN(string);
    LM_MALLOC(strlist, sizeof(*strlist));
    /* Always at least one line */
    strlist->num_lines = 1;
    strlist->lines = NULL;
    strlist->buffer = NULL;
    LM_MALLOC(strlist->buffer, LM_STRING_LEN(string) + 1);
    strncpy(strlist->buffer, LM_STRING_BUFFER(string),
            LM_STRING_LEN(string) + 1);
    LM_MEMASSERT(strlist->buffer);

    /* TODO Build line offset list in one pass instead of two */

    /* Find the number of lines in the string */
    for (i = 0; i < len; i++) {
        if (strlist->buffer[i] == '\n') {
            strlist->num_lines++;
        }
    }

    /* Allocate the buffer to hold the offsets */
    LM_MALLOC(strlist->lines, sizeof(*strlist->lines) * strlist->num_lines);
    /* Go through the list again and build the offset list */
    cur = 1;
    strlist->lines[0] = 0;
    for (i = 0; i < len; i++) {
        if (strlist->buffer[i] == '\n') {
            strlist->buffer[i] = '\0';
            strlist->lines[cur++] = i + 1;
            i++;
        }
    }

    /* If there was a newline at the end of the last line,
       remove that from the line list */
    if (strlist->buffer[strlist->lines[strlist->num_lines - 1]] == '\0') {
        strlist->num_lines--;
    }

    LM_MEMASSERT(strlist->lines);
    LM_MEMASSERT(strlist->buffer);

    *pstrlist = (strlist_t) strlist;

    return retval;
}

int strlist_free(strlist_t * pstrlist)
{
    int retval = 0;
    tag_strlist_t *strlist = NULL;

    strlist = (tag_strlist_t *) * pstrlist;

    LM_FREE(strlist->lines);
    LM_FREE(strlist->buffer);
    LM_FREE(strlist);
    *pstrlist = (strlist_t) strlist;
    return retval;
}

int strlist_string_cmp(lm_string_t a, lm_string_t b)
{
    return LM_STRING_CMP(a, b);
}

int strlist_regex_cmp(lm_string_t r, lm_string_t b)
{
    return LM_STRING_CMP(r, b);
}

int strlist_domain_cmp(lm_string_t domain, lm_string_t wcdomain, char wcchar)
{
    int match = 1;

    /* Check for a leading dot on the domain, which represents a wildcard */
    if (LM_STRING_BUFFER(wcdomain)[0] == wcchar) {

        lm_string_t tempdomain = NULL;
        lm_string_t tempwcdomain = NULL;

        /* If the domain is shorter than the wildcard domain without the 
         * leading period, we can't possibly have a match so we won't 
         * bother checking 
         */
        if (LM_STRING_LEN(domain) < LM_STRING_LEN(wcdomain) - 1) {
            return 1;
        }

        /* Make scratch copies of the domains */
        if (LM_STRING_DUP(tempdomain, domain)
            || LM_STRING_DUP(tempwcdomain, wcdomain)) {
            return -1;
        }

        /* Shift off the first dot in the domain */
        LM_STRING_START_SHIFT(tempwcdomain, 1);

        /* While there is a possibility of a match and there isn't a match... */
        while ((LM_STRING_LEN(tempdomain)) >= LM_STRING_LEN(tempwcdomain)
               && (match != 0)) {
            /* If the first character of the strings and lengths match */
            if ((LM_STRING_BUFFER(tempdomain)[0] ==
                 LM_STRING_BUFFER(tempwcdomain)[0])
                && (LM_STRING_LEN(tempdomain) == LM_STRING_LEN(tempdomain))) {
                /* See if the strings match */
                match = LM_STRING_CMP(tempwcdomain, tempdomain);
            }

            /* Shift off the first character in the string and
             * look again */
            LM_STRING_START_SHIFT(tempdomain, 1);
        }

        /* Free the strings */
        LM_STRING_FREE(tempdomain);
        LM_STRING_FREE(tempwcdomain);
    } else {
        /* If the strings are different length, don't bother  checking */
        if (LM_STRING_LEN(domain) != LM_STRING_LEN(wcdomain)) {
            return 1;
        }

        match = LM_STRING_CMP(wcdomain, domain);
    }

    return match;
}

int strlist_to_string(lm_string_t * pstr, strlist_t strlist)
{
    int retval = 0;
    tag_strlist_t *list;
    int cur;
    lm_string_t tmpstring = NULL;

    list = (tag_strlist_t *) strlist;
    LM_MEMASSERT(list->buffer);
    LM_MEMASSERT(list->lines);
    for (cur = 0; cur < list->num_lines; cur++) {
        list->buffer[list->lines[cur] - 1] = '\n';
    }

    LM_STRING_NEW(tmpstring, list->buffer);

    for (cur = 0; cur < list->num_lines; cur++) {
        list->buffer[list->lines[cur] - 1] = '\0';
    }

    *pstr = tmpstring;
    return retval;
}

int strlist_cat(strlist_t * plist, lm_string_t string)
{
    int retval = 0;
    tag_strlist_t *list;
    strlist_t newlist = NULL;
    int cur;
    lm_string_t tmpstring = NULL;

    list = (tag_strlist_t *) * plist;
    LM_MEMASSERT(list->buffer);
    LM_MEMASSERT(list->lines);
    for (cur = 0; cur < list->num_lines; cur++) {
        list->buffer[list->lines[cur] - 1] = '\n';
    }

    if (LM_STRING_NEW(tmpstring, list->buffer)
        || LM_STRING_CAT(tmpstring, string) || STRLIST_NEW(newlist, tmpstring)
        || LM_STRING_FREE(tmpstring)) {
        retval = -1;
    } else {

        STRLIST_FREE(*plist);
        *plist = newlist;

        list = (tag_strlist_t *) newlist;
        LM_MEMASSERT(list->lines);
        LM_MEMASSERT(list->buffer);
    }

    return retval;
}

int strlist_atdomain_cmp(lm_string_t domain, lm_string_t wcdomain)
{
    return strlist_domain_cmp(domain, wcdomain, '@');
}

int strlist_dotdomain_cmp(lm_string_t domain, lm_string_t wcdomain)
{
    return strlist_domain_cmp(domain, wcdomain, '.');
}

int strlist_search(strlist_t strlist, lm_string_t str,
                   int (*cmp) (lm_string_t, lm_string_t))
{
    int cur;
    tag_strlist_t *list;
    int found = 1;

    if (strlist == NULL) {
        /* 1 = not found */
        return (1);
    }

    /* Get a referencable pointer to the list */
    list = (tag_strlist_t *) strlist;

    LM_MEMASSERT(list->lines);
    LM_MEMASSERT(list->buffer);

    /* Empty list, can't be in it */
    if (list->num_lines == 0) {
        return found;
    }

    /* For each string in the list */
    for (cur = 0; cur < list->num_lines && found != 0; cur++) {
        lm_string_t line = NULL;

        LM_STRING_NEW(line, list->buffer + list->lines[cur]);
        if (str != NULL) {
            if (cmp(str, line) == 0) {
                found = 0;
            }
        }
        LM_STRING_FREE(line);
    }

    LM_MEMASSERT(list->lines);
    LM_MEMASSERT(list->buffer);

    return found;
}
