/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: spam_check.c,v 1.33 2003/09/24 21:10:36 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

#include <string.h>
#include <liblm.h>

#include "magic-smtpd.h"
#include "config_msd.h"
#include "config_qmail.h"
#include "spam_check.h"
#include "smtp_check.h"
#include "strlist.h"
#include "utils.h"
#include "common/spam_log.h"

#include "common/controls.h"
#include "common/load_controls.h"

extern config_t config;
extern config_type_t config_type_info[];

extern int fakehelohost;
extern lm_string_t helohost;



/* this is global defined in magic-smtpd.c */
extern int mail_from_rfc_formatted;

static lm_string_t fromuser = NULL;
static lm_string_t fromdomain = NULL;
static lm_string_t rcptuser = NULL;
static lm_string_t rcptdomain = NULL;
static lm_string_t fromcountry = NULL;
static lm_string_t helocountry = NULL;
static lm_string_t remoteipcountry = NULL;

int spam_check_init(config_spam_rule_t * rules, lm_string_t rcptaddr,
                    lm_string_t fromaddr, lm_string_t helodomain)
{
    int retval = 0;
    int index;

    if (get_user_domain(&rcptuser, &rcptdomain, rcptaddr)
        || get_user_domain(&fromuser, &fromdomain, fromaddr)) {
        retval = -1;
    }

    /* get the last part of the domain, which is possibly a country code */
    if (!LM_STRING_RCHR(fromdomain, '.', index)) {
        if (LM_STRING_NEW
            (fromcountry, LM_STRING_BUFFER(fromdomain) + index + 1)) {
            retval = 1;
        }
    } else {
        LM_STRING_NEW(fromcountry, "");
    }

    if (LM_STRING_LEN(helodomain) > 0) {
        /* get the last part of the domain, which is possibly a country code */
        if (!LM_STRING_RCHR(helodomain, '.', index)) {
            if (LM_STRING_NEW
                (helocountry, LM_STRING_BUFFER(helodomain) + index + 1)) {
                retval = 1;
            }                
        }
    } else {
        LM_STRING_NEW(helocountry, "");
    }

    if (ip_to_country_code(&remoteipcountry, config_qmail.tcpremoteip))
        LM_STRING_NEW(remoteipcountry, "");
    /* If the country returned was not known ** is returned by the program. 
       Since we don't want to search the country list when this happens
       just turn that into an empty string so the search will automatically 
       succeed */
    if (LM_STRING_BUFFER(remoteipcountry)[0] == '*'
        && LM_STRING_BUFFER(remoteipcountry)[1] == '*') {
        LM_STRING_START_SHIFT(remoteipcountry, LM_STRING_LEN(remoteipcountry));
    }

    return retval;
}

void spam_check_cleanup(void)
{
    if (rcptuser) {
        LM_STRING_FREE(rcptuser);
    }
    if (rcptdomain) {
        LM_STRING_FREE(rcptdomain);
    }
    if (fromuser) {
        LM_STRING_FREE(fromuser);
    }
    if (fromdomain) {
        LM_STRING_FREE(fromdomain);
    }
    if (fromcountry) {
        LM_STRING_FREE(fromcountry);
    }
    if (helocountry) {
        LM_STRING_FREE(helocountry);
    }
    if (remoteipcountry) {
        LM_STRING_FREE(remoteipcountry);
    }
}

/* Empty functions for upper level spam rules */
int all_global_rules(config_spam_rule_t * rules)
{
    return 0;
}

int spam_check(config_spam_rule_t * rules)
{
    return (rules->spam_check == 0);
}

int smtp_check(config_spam_rule_t * rules)
{
    return (rules->smtp_check == 0);
}

int spam_check_do(config_spam_rule_t * rules, lm_string_t rcptaddr,
                  lm_string_t fromaddr, int *whitelist)
{
    int retval = 0;
    char hit_str[1024];
    char buffer[1024];

    /* not found on whitelist yet */
    *whitelist = 0;

    /* no hits yet */
    hit_str[0] = '\0';

    LM_ASSERT(rules != NULL);
    LM_STRING_ASSERT(rcptaddr);
    LM_STRING_ASSERT(fromaddr);
    retval |= spam_check_init(rules, rcptaddr, fromaddr, helohost);

#define NODEPS (1)
#define CHECK_RULE(RULE, DEPENDS, ARGS, MSG) \
if((retval == 0) && \
   (rules->spam_check != 0) && \
   (rules->RULE != 0) && \
   (DEPENDS)) \
     { \
       syslog(LOG_DEBUG, "checking rule: %s", #RULE);   \
       if((retval |= RULE ARGS != 0)) \
         { \
            addSpamLog(#RULE, NULL, 1, 0); \
      	    snprintf(hit_str, 1024, "SPAM_HIT %s R:%s F:%s %s", \
		   LM_STRING_BUFFER(config_qmail.tcpremoteip), \
		   LM_STRING_BUFFER(rcptaddr), LM_STRING_BUFFER(fromaddr), \
		   #RULE); \
	 } \
     }

    /* Perform checks based on smtp information */
    /* NOTE: Descriptive message is no longer used in log output */
    CHECK_RULE(all_global_rules, NODEPS, (rules),
               "All global spam rules inherited");
    CHECK_RULE(spam_check, NODEPS, (rules),
               "Spam checking not enabled by the user");
    CHECK_RULE(smtp_check, (rules->spam_check), (rules),
               "SMTP Spam checking not enabled.");
    CHECK_RULE(require_full_addr, (rules->spam_check && rules->smtp_check),
               (rcptaddr), "Full address not given to RCPT (user@host).");
    CHECK_RULE(block_mail_from_self, (rules->spam_check && rules->smtp_check),
               (rcptaddr, fromaddr),
               "Address in MAIL From: identical to RCPT To:!");
    CHECK_RULE(block_ip_in_addr, (rules->spam_check && rules->smtp_check),
               (rcptdomain), "Reciept address contains an IP address!");
    CHECK_RULE(block_ip_in_addr,
               (rules->spam_check && rules->smtp_check
                && (LM_STRING_LEN(fromaddr) > 0)), (fromdomain),
               "From address contains an IP address!");

    /* only do the following check if the fromaddr is not 0 (which would indicate that it was a bounce) */
    if ((LM_STRING_LEN(fromaddr) == 4)
        && (strcmp(LM_STRING_BUFFER(fromaddr), "#@[]") == 0)) {

        /* Special case, qmail uses #@[] as the double bounce MAIL FROM address */
        /* we do not want to check the domain on these messages */

        SYSLOG((LOG_INFO, "matched #@[], ignoring MAIL FROM checks."));

    } else {

        CHECK_RULE(valid_from_domain,
                   (rules->spam_check && rules->smtp_check
                    && (LM_STRING_LEN(fromaddr) > 0)), (fromdomain),
                   "From: domain does not resolve!");

        CHECK_RULE(valid_bounce, (rules->spam_check && rules->smtp_check),
                   (fromdomain), "Bounce to from domain will not work!");

        CHECK_RULE(mail_from_strict_addr_parse,
                   (rules->spam_check
                    && rules->smtp_check), (fromaddr, mail_from_rfc_formatted),
                   "Mail from is not RFC compliant");
    }

    CHECK_RULE(require_helo, (rules->spam_check && rules->smtp_check),
               (helohost), "Server did not provide HELO.");
    CHECK_RULE(valid_helo_domain,
               (rules->spam_check && rules->smtp_check
                && rules->require_helo), (helohost),
               "HELO domain does not resolve!");
    CHECK_RULE(check_ip_reverse_dns, (rules->spam_check && rules->smtp_check),
               (config_qmail.tcpremoteip),
               "Remote IP does not have an associated domain name!");

#define CHECK_IN_LIST(STRING, DEPENDS, LIST, CMPFUNC, MSG) \
  if((rules->spam_check != 0) && \
     (DEPENDS)) \
    { \
      int in_whitelist = 0; \
      int in_global_whitelist = 0; \
      int in_blacklist = 0; \
      /* Search global whitelist if requested */ \
      if(rules->use_global_##LIST##_whitelist) { \
        syslog(LOG_DEBUG, "checking list: global_%s_whitelist", #LIST);   \
	    in_global_whitelist = in_whitelist = !strlist_search(global_spam_rules.LIST##_whitelist, STRING, CMPFUNC); \
      } \
      /* Search personal whitelist */ \
      if(!in_whitelist) { \
        syslog(LOG_DEBUG, "checking list: %s_whitelist", #LIST);   \
	    in_whitelist = !strlist_search(rules->LIST##_whitelist, STRING, CMPFUNC); \
      } \
      if (in_whitelist) { \
          /* if it is in whitelist, it is not spam */ \
           snprintf(buffer, 1024, "%s_whitelist", #LIST); \
           addSpamLog(buffer, NULL, -1, in_global_whitelist); \
	       SYSLOG((LOG_NOTICE, "WHITELIST: %s", #LIST)); \
          *whitelist = 1; \
          return (0); \
      } else \
	    { \
	    /* Search global blacklist if requested */ \
	    if(rules->use_global_##LIST##_blacklist) { \
            syslog(LOG_DEBUG, "checking list: global_%s_blacklist", #LIST);   \
	        in_blacklist = !strlist_search(global_spam_rules.LIST##_blacklist, STRING, CMPFUNC); \
        } \
	    if(in_blacklist) \
	        { \
            snprintf(buffer, 1024, "%s_blacklist", #LIST); \
            addSpamLog(buffer, NULL, 1, 1); \
	        snprintf(hit_str, 1024, "SPAM_HIT %s R:%s F:%s %s_blacklist GLOBAL", \
		    LM_STRING_BUFFER(config_qmail.tcpremoteip), \
		    LM_STRING_BUFFER(rcptaddr), LM_STRING_BUFFER(fromaddr), \
		    #LIST); \
	        } \
	    else \
	        { \
	        /* Search personal whitelist */ \
	        if(!in_blacklist) { \
                syslog(LOG_DEBUG, "checking list: %s_blacklist", #LIST);   \
		        in_blacklist = !strlist_search(rules->LIST##_blacklist, STRING, CMPFUNC); \
            } \
	        if(in_blacklist) \
		        { \
                snprintf(buffer, 1024, "%s_blacklist", #LIST); \
                addSpamLog(buffer, NULL, 1, 0); \
	            snprintf(hit_str, 1024, "SPAM_HIT %s R:%s F:%s %s_blacklist", \
		        LM_STRING_BUFFER(config_qmail.tcpremoteip), \
		        LM_STRING_BUFFER(rcptaddr), LM_STRING_BUFFER(fromaddr), \
		        #LIST); \
		        } \
	        } \
	    } \
      retval |= !in_whitelist && in_blacklist; \
    }

#define STRING_IN_LIST(STRING, DEPENDS, LIST, MSG) \
CHECK_IN_LIST(STRING, DEPENDS, LIST, strlist_string_cmp, MSG)
#define REGEX_IN_LIST(REGEX, DEPENDS, LIST, MSG) \
CHECK_IN_LIST(REGEX, DEPENDS, LIST, strlist_regex_cmp, MSG)
#define DOMAIN_IN_LIST(DOMAIN, DEPENDS, LIST, MSG) \
CHECK_IN_LIST(DOMAIN, DEPENDS, LIST, strlist_atdomain_cmp, MSG)

    STRING_IN_LIST(helohost,
                   (rules->spam_check && rules->smtp_check
                    && rules->valid_helo_domain
                    && rules->require_helo), helo, "HELO domain in blacklist");
    STRING_IN_LIST(helocountry,
                   (rules->spam_check && rules->smtp_check
                    && rules->valid_helo_domain
                    && rules->require_helo), country,
                   "HELO country in blacklist");
    DOMAIN_IN_LIST(fromaddr,
                   (rules->spam_check && rules->smtp_check
                    && rules->valid_from_domain), from,
                   "From: address in blacklist");
    STRING_IN_LIST(fromcountry, (rules->spam_check && rules->smtp_check),
                   country, "From: country in blacklist");
    STRING_IN_LIST(config_qmail.tcpremoteip,
                   (rules->spam_check && rules->smtp_check
                    && rules->require_helo), ip,
                   "Remote Server IP address in blacklist");
    STRING_IN_LIST(remoteipcountry, (rules->smtp_check && rules->smtp_check),
                   country, "IP source country in country blacklist");

    if ((*whitelist == 0) && (retval != 0) && (hit_str[0] != '\0')) {
        SYSLOG((LOG_NOTICE, "%s", hit_str));
    }

    spam_check_cleanup();

    return retval;
}
