
/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: smtp_check.c,v 1.18 2003/09/24 21:10:36 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

/* C includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* System includes */
#include <sys/types.h>
#include <unistd.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <resolv.h>
#include <sys/socket.h>

/* lmlibs */
#include <liblm.h>

/* magic-smtpd includes */
#include "smtp_check.h"
#include "config_msd.h"
#include "utils.h"
#include "getmx.h"
#include "common/controls.h"
#include "common/load_controls.h"

extern config_t config;
extern config_type_t config_type_info[];


int require_helo(lm_string_t helostring)
{
    int retval = 0;
    char *buffer;

    buffer = LM_STRING_BUFFER(helostring);

    if (buffer[0] == 'u' && buffer[1] == 'n' && buffer[2] == 'k'
        && buffer[3] == 'n' && buffer[4] == 'o' && buffer[5] == 'w'
        && buffer[6] == 'n') {
        retval = -1;
    }
    return retval;
}

int smtp_check_valid_domain(lm_string_t domain)
{
    char buffer[1024];
    int retval = 0;

    /* If given an ip address don't bother resolving and say it's valid */
    if (!is_ip(domain)) {
        SYSLOG((LOG_INFO, "Did not attempt to resolve IP address"));
        return 0;
    }

    SYSLOG((LOG_INFO, "Lookup up domain %s (this may take a while)",
            LM_STRING_BUFFER(domain)));

    retval = res_query(LM_STRING_BUFFER(domain), C_IN, T_A, buffer, sizeof(buffer));
    if (retval == -1) {
        retval = res_query(LM_STRING_BUFFER(domain), C_IN, T_MX, buffer, sizeof(buffer));
    }
    return retval == -1;
}

int valid_helo_domain(lm_string_t domain)
{
    /* default fail return value since zero length domain wont resolve */
    int retval = -1;

    if (LM_STRING_LEN(domain) > 0) {
        retval = smtp_check_valid_domain(domain);
    }
    return retval;
}

int valid_from_domain(lm_string_t domain)
{
    /* default fail return value since zero length domain wont resolve */
    int retval = -1;

    if (LM_STRING_LEN(domain) > 0) {
        retval = smtp_check_valid_domain(domain);
    }
    return retval;
}

int block_mail_from_self(lm_string_t from, lm_string_t to)
{
    int retval;

    /* Very basic check to compare the two addresses */
    retval = !LM_STRING_CMP(from, to);
    return retval;
}

int mail_from_strict_addr_parse(lm_string_t addr, int rfc)
{
    int rfa;

    if ((LM_STRING_LEN(addr) == 0) && (rfc == 1)) {
        /* MAIL FROM: <>
         * so it was a bounce message 
         * so we accept it */
        return (0);
    }

    rfa = require_full_addr(addr);

    if ((rfc == 1) && (rfa == 0)) {
        /* this is a full address, and the rfc flag is set */

        /* success: passes strict parsing */
        return (0);
    }

    /* failure: did not live up to our strict address */
    return (1);
}

int require_full_addr(lm_string_t addr)
{
    int retval = 1;
    int index;

    /* If we have an @ sign and it wasn't
       the first or last character */
    if (!LM_STRING_CHR(addr, '@', index) && (index != 0)
        && LM_STRING_LEN(addr) != (index - 1)) {
        retval = 0;
    }
    return retval;
}

int smtp_blocking(void)
{
    return 0;
}

int valid_bounce(lm_string_t fromdomain)
{
    struct sockaddr_in serv_addr;
    mxdata_t mxdata;
    struct hostent *hostinfo;
    int i;

    if (is_ip(fromdomain) != 0) {

        /* is not an ip address */

        mxdata = mx_query(LM_STRING_BUFFER(fromdomain));
        if (mxdata == NULL) {
            return (-1);
        }

        /* printf("FOUND %d MX RECORDS\n", mxdata->mxcount); */
        for (i = 0; i < mxdata->mxcount; i++) {
            int sock;

            serv_addr.sin_family = AF_INET;
            serv_addr.sin_port = htons(25);

            if (mxdata->mxhost[i]->addrcount > 0) {
                serv_addr.sin_addr.s_addr = htonl(mxdata->mxhost[i]->addr[0]);
            } else {
                /* need to look up IP */
                hostinfo = gethostbyname(mxdata->mxhost[i]->name);
                if (hostinfo != NULL) {
                    serv_addr.sin_addr = *(struct in_addr *) hostinfo->h_addr;
                } else {
                    /* can't lookup mx */
                    SYSLOG((LOG_INFO, "FAILED TO RESOLVE MX '%s'",
                            mxdata->mxhost[i]->name));
                    continue;   /* for (... */
                }
            }

            sock = socket(AF_INET, SOCK_STREAM, 0);
            if (!connect
                (sock, (struct sockaddr *) &serv_addr,
                 sizeof(struct sockaddr))) {
                /* we could connect so close socket */
                SYSLOG((LOG_INFO, "Successfull bounce check connection to '%s'",
                        mxdata->mxhost[i]->name));
                close(sock);
                free(mxdata);
                return (0);
            } else {
                SYSLOG((LOG_INFO,
                        "Could not connect to port 25 on MX hostname '%s'",
                        mxdata->mxhost[i]->name));
            }
        }

        mx_free(mxdata);
    }

    return (-1);
}

int block_ip_in_addr(lm_string_t addr)
{
    int retval;

    retval = !is_ip(addr);
    return retval;
}

int check_ip_reverse_dns(lm_string_t ip)
{
    struct hostent *host = NULL;
    struct in_addr inp;
    int retval = 0;

    /* If this is not a d.d.d.d type string don't lookup */
    if (is_ip(ip)) {
        return 0;
    }

    if (inet_aton(LM_STRING_BUFFER(ip), &inp) == 0) {
        /* address is invalid, fail open */
        return (0);
    }

    /* 
     * Attempt to do a reverse lookup on IP. If gethostbyname() fails with 
     * TRY_AGAIN, try again until it either gives us a real failure or succeeds. 
     */
    while (((host = gethostbyaddr(&inp, sizeof(inp), AF_INET)) == NULL)
           && (h_errno == TRY_AGAIN)) {
        ;                       /* do nothing */
    }

    /* Check for error */
    if (host == NULL) {
        switch (h_errno) {
            case HOST_NOT_FOUND:
                syslog(LOG_DEBUG, "Hostname for IP %s could not be found",
                       LM_STRING_BUFFER(ip));
                retval = 1;
                break;
            case NO_RECOVERY:
                syslog(LOG_ERR, "A non-recoverable DNS server error occured.");
                retval = -1;
                break;
            default:
                /* everything ok, return sucess */
                retval = 0;
        }
    }
    return retval;
}
