/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: smtp.c,v 1.36 2003/10/23 21:08:50 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <unistd.h>

#include <liblm.h>

#include "config_msd.h"
#include "smtp.h"

#include "common/controls.h"
#include "common/load_controls.h"

extern config_t config;
extern config_type_t config_type_info[];


#define SMTP_LOG_SYS "smtp"
#define SMTP_OUTBUF_SIZE 1024
static char outbuf[SMTP_OUTBUF_SIZE];

typedef struct {
    char *command;
    int (*handler) (lm_string_t args);
} smtp_command_t;

/* this is defined in magic-smtp.c */
extern int mail_from_rfc_formatted;

static int smtp_HELO_handler(lm_string_t args);
static int smtp_EHLO_handler(lm_string_t args);
static int smtp_MAIL_handler(lm_string_t args);
static int smtp_RCPT_handler(lm_string_t args);
static int smtp_DATA_handler(lm_string_t args);
static int smtp_AUTH_handler(lm_string_t args);
static int smtp_QUIT_handler(lm_string_t args);
static int smtp_RSET_handler(lm_string_t args);
static int smtp_HELP_handler(lm_string_t args);
static int smtp_NOOP_handler(lm_string_t args);
static int smtp_VRFY_handler(lm_string_t args);
static int smtp_rset(void);

static int gotmail_flag;
static int gotrcpt_flag;
static int timeout;

smtp_command_t smtp_commands[] = {
    {"HELO", smtp_HELO_handler},
    {"EHLO", smtp_EHLO_handler},
    {"MAIL", smtp_MAIL_handler},
    {"RCPT", smtp_RCPT_handler},
    {"DATA", smtp_DATA_handler},
    {"AUTH", smtp_AUTH_handler},
    {"QUIT", smtp_QUIT_handler},
    {"RSET", smtp_RSET_handler},
    {"HELP", smtp_HELP_handler},
    {"NOOP", smtp_NOOP_handler},
    {"VRFY", smtp_VRFY_handler},
};
int num_smtp_commands = sizeof(smtp_commands) / sizeof(smtp_commands[0]);

/* Default callback functions or protoypts for those that won't fit on one line */
static int smtp_helo_default_callback(lm_string_t domain);
static int smtp_ehlo_default_callback(lm_string_t domain);
static int smtp_mail_default_callback(lm_string_t addr);
static int smtp_rcpt_default_callback(lm_string_t addr);
static int smtp_data_default_callback(void);
static int smtp_auth_default_callback(int authtype, lm_string_t username64);
static int smtp_rset_default_callback(void);
static int smtp_help_default_callback(void);
static int smtp_noop_default_callback(void);
static int smtp_vrfy_default_callback(lm_string_t addr);
static int smtp_quit_default_callback(int exitcode);
static int smtp_read_default_callback(int fd, lm_string_t * string);
static int smtp_write_default_callback(int fd, lm_string_t string);
static int smtp_queue_init_default_callback(void);
static int smtp_queue_default_callback(void);
static int smtp_queue_cleanup_default_callback(void);
static int smtp_data_line_default_callback(lm_string_t line);

/* Function pointer variables for the various command callback functions */
static smtp_read_callback_t smtp_read_callback = smtp_read_default_callback;
static smtp_write_callback_t smtp_write_callback = smtp_write_default_callback;
static smtp_queue_init_callback_t smtp_queue_init_callback =
    smtp_queue_init_default_callback;
static smtp_queue_callback_t smtp_queue_callback = smtp_queue_default_callback;
static smtp_queue_cleanup_callback_t smtp_queue_cleanup_callback =
    smtp_queue_cleanup_default_callback;
static smtp_helo_handler_callback_t smtp_helo_callback =
    smtp_helo_default_callback;
static smtp_ehlo_handler_callback_t smtp_ehlo_callback =
    smtp_ehlo_default_callback;
static smtp_mail_handler_callback_t smtp_mail_callback =
    smtp_mail_default_callback;
static smtp_rcpt_handler_callback_t smtp_rcpt_callback =
    smtp_rcpt_default_callback;
static smtp_data_handler_callback_t smtp_data_callback =
    smtp_data_default_callback;
static smtp_auth_handler_callback_t smtp_auth_callback =
    smtp_auth_default_callback;
static smtp_quit_handler_callback_t smtp_quit_callback =
    smtp_quit_default_callback;
static smtp_rset_handler_callback_t smtp_rset_callback =
    smtp_rset_default_callback;
static smtp_help_handler_callback_t smtp_help_callback =
    smtp_help_default_callback;
static smtp_noop_handler_callback_t smtp_noop_callback =
    smtp_noop_default_callback;
static smtp_vrfy_handler_callback_t smtp_vrfy_callback =
    smtp_vrfy_default_callback;
static smtp_data_line_callback_t smtp_header_line_callback =
    smtp_data_line_default_callback;
static smtp_data_line_callback_t smtp_body_line_callback =
    smtp_data_line_default_callback;

#define smtp_printf(fmt) sprintf fmt; { lm_string_t tmpstr = NULL; LM_STRING_NEW(tmpstr, outbuf); smtp_write_callback(1, tmpstr); LM_STRING_FREE(tmpstr); }
static void smtp_error_out_of_memory(void)
{
    smtp_printf((outbuf, "421 out of memory (#4.3.0)\r\n"));
    smtp_quit_callback(1);
};

/* Intialize things and setup the callback functions for each command */
int smtp_init(int smtp_timeout, smtp_read_callback_t pread_func,
              smtp_write_callback_t pwrite_func,
              smtp_queue_init_callback_t pqueue_init_func,
              smtp_queue_callback_t pqueue_func,
              smtp_queue_cleanup_callback_t pqueue_cleanup_func,
              smtp_helo_handler_callback_t phelo_func,
              smtp_ehlo_handler_callback_t pehlo_func,
              smtp_mail_handler_callback_t pmail_func,
              smtp_rcpt_handler_callback_t prcpt_func,
              smtp_data_handler_callback_t pdata_func,
              smtp_auth_handler_callback_t pauth_func,
              smtp_vrfy_handler_callback_t pvrfy_func,
              smtp_rset_handler_callback_t prset_func,
              smtp_help_handler_callback_t phelp_func,
              smtp_noop_handler_callback_t pnoop_func,
              smtp_quit_handler_callback_t pquit_func,
              smtp_data_line_callback_t pheader_line_func,
              smtp_data_line_callback_t pbody_line_func)
{
    /* Setup the handlers if they were passed as parameters */
    if (pread_func != NULL) {
        smtp_read_callback = pread_func;
    }
    if (pwrite_func != NULL) {
        smtp_write_callback = pwrite_func;
    }
    if (pqueue_init_func != NULL) {
        smtp_queue_init_callback = pqueue_init_func;
    }
    if (pqueue_func != NULL) {
        smtp_queue_callback = pqueue_func;
    }
    if (pqueue_cleanup_func != NULL) {
        smtp_queue_cleanup_callback = pqueue_cleanup_func;
    }
    if (phelo_func != NULL) {
        smtp_helo_callback = phelo_func;
    }
    if (pehlo_func != NULL) {
        smtp_ehlo_callback = pehlo_func;
    }
    if (pmail_func != NULL) {
        smtp_mail_callback = pmail_func;
    }
    if (prcpt_func != NULL) {
        smtp_rcpt_callback = prcpt_func;
    }
    if (pdata_func != NULL) {
        smtp_data_callback = pdata_func;
    }
    if (pauth_func != NULL) {
        smtp_auth_callback = pauth_func;
    }
    if (pquit_func != NULL) {
        smtp_quit_callback = pquit_func;
    }
    if (prset_func != NULL) {
        smtp_rset_callback = prset_func;
    }
    if (phelp_func != NULL) {
        smtp_help_callback = phelp_func;
    }
    if (pnoop_func != NULL) {
        smtp_noop_callback = pnoop_func;
    }
    if (pvrfy_func != NULL) {
        smtp_vrfy_callback = pvrfy_func;
    }
    if (pheader_line_func != NULL) {
        smtp_header_line_callback = pheader_line_func;
    }
    if (pbody_line_func != NULL) {
        smtp_body_line_callback = pbody_line_func;
    }

    timeout = smtp_timeout;

    return 0;
}

static int smtp_read_default_callback(int fd, lm_string_t * string)
{
    lm_string_t newstr = NULL;
    int bytesread;

    if (LM_STRING_FGETS(newstr, stdin, SMTP_MAX_LINE_LENGTH, bytesread)) {
        return -1;
    }

    LM_WARN(*string != NULL);
    *string = newstr;
    return *string == NULL;
}

static int smtp_write_default_callback(int fd, lm_string_t string)
{
    fprintf(stdout, "%s", LM_STRING_BUFFER(string));
    fflush(stdout);

    return 0;
}

static int smtp_queue_init_default_callback(void)
{
    SYSLOG((LOG_INFO,
            "default smtp queue init function called -- doing nothing"));
    /* Do nothing for now */

    return 0;
}

static int smtp_queue_default_callback(void)
{
    SYSLOG((LOG_INFO, "default smtp queue function called -- doing nothing"));
    /* Do nothing for now */

    return 0;
}

static int smtp_queue_cleanup_default_callback(void)
{
    SYSLOG((LOG_INFO,
            "default smtp queue cleanup function called -- doing nothing"));
    /* Do nothing for now */

    return 0;
}

static int smtp_data_line_default_callback(lm_string_t line)
{
    SYSLOG((LOG_INFO, "default data line callback called"));
    /* Do nothing for now */

    return 0;
}

/* Default function called by QUIT command if none is given to smtp_init */
static int smtp_quit_default_callback(int exitcode)
{
    SYSLOG((LOG_INFO, "QUIT default callback function called, exitcode: %d",
            exitcode));
    fflush(stdout);
    fflush(stderr);
    smtp_cleanup();
    exit(exitcode);

    /* gets rid of warning for not returning value */

    return exitcode;
}

/* Default callback functions or protoypts for those that won't fit on one line */
static int smtp_auth_default_callback(int authtype, lm_string_t username64)
{
    SYSLOG((LOG_INFO, "AUTH default callback function called, authtype %d",
            authtype));

    return 0;
}

static int smtp_rset_default_callback(void)
{
    SYSLOG((LOG_INFO, "RSET default callback function called"));

    return 0;
}

static int smtp_help_default_callback(void)
{
    SYSLOG((LOG_INFO, "HELP default callback function called"));

    return 0;
}

static int smtp_noop_default_callback(void)
{
    SYSLOG((LOG_INFO, "NOOP default callback function called"));

    return 0;
}

static int smtp_vrfy_default_callback(lm_string_t addr)
{
    SYSLOG((LOG_INFO, "VRFY default callback function called, addr : %s",
            LM_STRING_BUFFER(addr)));
    return 0;
}

/* Read a command from stdin into *command */
int smtp_read_command(lm_string_t * pcommand)
{
    lm_string_t string = NULL;

    /* Set an alarm for each command as noted in RFC */
    alarm(timeout);

    if (smtp_read_callback(0, &string)) {
        SYSLOG((LOG_INFO, "Command line too long"));
        printf("500 command line too long (#4.2.2.)\r\n");
        LM_STRING_NEW(string, "");
        *pcommand = string;
        return 1;
    }

    LM_STRING_ASSERT(string);
    *pcommand = string;

    return *pcommand == NULL;
}

/* Parse the command and call the appropriate handler for the given command,
   else print "unimplemented" and return */
int smtp_handle_command(lm_string_t command)
{
    int i;
    char *buffer;
    int invalid_cmd_flag = 0;
    char *cmd;

    LM_STRING_ASSERT(command);

    /* Get rid of the whitespace on the head and tail of the command */
    LM_STRING_CHOP_HEAD(command);
    LM_STRING_CHOP(command);

    /* Don't process blank lines */
    if (LM_STRING_LEN(command) < 4) {
        invalid_cmd_flag = 1;
    }

    /* Get a pointer to the string */
    buffer = LM_STRING_BUFFER(command);

    /* All smtp commands must be 4 chars long */
    if (!invalid_cmd_flag && buffer[4] != ' ' && buffer[4] != '\t'
        && buffer[4] != '\0') {
        invalid_cmd_flag = 1;
    }

    for (i = 0; !invalid_cmd_flag && i < num_smtp_commands; i++) {
        cmd = smtp_commands[i].command;

        if (cmd[0] == toupper(buffer[0]) && cmd[1] == toupper(buffer[1])
            && cmd[2] == toupper(buffer[2]) && cmd[3] == toupper(buffer[3])) {
            lm_string_t args = NULL;
            int retval;

            /* Get the arguments for the command and chop off leading whitespace */
            if (LM_STRING_NEW(args, &buffer[4])) {
                smtp_error_out_of_memory();
                return 0;
            }
            LM_STRING_CHOP_HEAD(args);
            LM_STRING_CHOP(args);

            /* Call the command handler with the given command arguments */
            retval = smtp_commands[i].handler(args);
            fflush(stdout);
            fflush(stderr);

            /* Cleanup argument string */
            LM_STRING_FREE(args);

            /* Return the value returned by the handler */
            return retval;
        }
    }

    if (invalid_cmd_flag || i == num_smtp_commands) {
        if (isascii(buffer[0]) && isascii(buffer[1]) && isascii(buffer[2])
            && isascii(buffer[3])) {
            SYSLOG((LOG_INFO, "Unimplmented command: %c%c%c%c", buffer[0],
                    buffer[1], buffer[2], buffer[3]));
        } else {
            SYSLOG((LOG_INFO, "Unimplmented command: unprintable"));
        }

        smtp_printf((outbuf, "502 unimplemented (#5.5.1)\r\n"));
    }

    /* No command was found so return error */
    return 0;
}

/* Find the address in the given string containing the reverse route list
   from the sender */
/* Returns 0 if the address was parsed sucessfully, non-zero on parse error */
static int smtp_parse_address(lm_string_t string, lm_string_t * paddr,
                              int *rfc_format)
{
    lm_string_t addr;
    int sindex, eindex;

    /* not rfc format unless it is */
    *rfc_format = 0;

    /* No arguements given to command */
    if (LM_STRING_LEN(string) == 0) {
        SYSLOG((LOG_WARNING, "bad address syntax, 0 length"));
        return -1;
    }

    sindex = eindex = -1;
    /* Handle From: First Last <user@host> addresses */
    if (!LM_STRING_CHR(string, '<', sindex)) {
        LM_STRING_START_SHIFT(string, sindex + 1);
        if (!LM_STRING_CHR(string, '>', eindex)) {
            LM_STRING_END_SHIFT(string, LM_STRING_LEN(string) - eindex);

            /* had a < and > so it's rfc formatted */
            *rfc_format = 1;
        }
    } else {
        /* Configuration option to only accept legal rfc addresses
           <user@host> */

        if (!config.rfc_addr_only) {
            /* Handle non <> bracketed address by searching for :, 
               and shifting it off */
            if (!LM_STRING_CHR(string, ':', sindex)) {
                LM_STRING_START_SHIFT(string, sindex + 1);
            }
            LM_STRING_CHOP_HEAD(string);

            /* strip off everything after the first space */
            if (!LM_STRING_CHR(string, ' ', eindex)) {
                LM_STRING_END_SHIFT(string, LM_STRING_LEN(string) - eindex);
            }

        } else {
            /* Bad address, no <>, return error */
            SYSLOG((LOG_WARNING, "bad address syntax, no <>"));
            return -1;
        }
    }

    /* If we have source route info, strip it off */
    if (LM_STRING_BUFFER(string)[0] == '@') {
        if (!LM_STRING_CHR(string, ':', sindex)) {
            SYSLOG((LOG_INFO, "Stripping source route from address"));
            LM_STRING_START_SHIFT(string, sindex + 1);
        }
    }

    /* Strip trailing white space */
    LM_STRING_CHOP(string);
    LM_STRING_CHOP_HEAD(string);

    /* QMAIL if the string is > 900 chars, return error */
    if (LM_STRING_LEN(string) > 900) {
        SYSLOG((LOG_WARNING, "address exceeds 900 characters in length"));
        LM_STRING_FREE(string);
        return -1;
    }

    /* Return the new string to the caller */
    addr = NULL;
    if (LM_STRING_NEW(addr, LM_STRING_BUFFER(string))) {
        smtp_error_out_of_memory();
        return 0;
    }

    LM_STRING_LOWER(addr);

    *paddr = addr;

    return 0;
}

/*** Handler function definitions */

static int smtp_HELO_handler(lm_string_t args)
{
    int retval;

    SYSLOG((LOG_INFO, "HELO command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* Call the helo callback handler function */
    retval = smtp_helo_callback(args);
    if (retval != 0) {
        return smtp_rset();
    }

    smtp_printf((outbuf, "250 %s \r\n", LM_STRING_BUFFER(args)));

    return smtp_rset();
}

static int smtp_helo_default_callback(lm_string_t domain)
{
    SYSLOG((LOG_INFO, "HELO default callback function called, domain: %s",
            LM_STRING_BUFFER(domain)));
    return 0;
}

static int smtp_EHLO_handler(lm_string_t args)
{
    int retval;

    SYSLOG((LOG_INFO, "EHLO command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* Call the ehlo callback handler function */
    retval = smtp_ehlo_callback(args);
    if (retval != 0) {
        return smtp_rset();
    }

    return smtp_rset();
}

static int smtp_ehlo_default_callback(lm_string_t domain)
{
    SYSLOG((LOG_INFO, "EHLO default callback function called, domain : %s",
            LM_STRING_BUFFER(domain)));
    return 0;
}

static int smtp_MAIL_handler(lm_string_t args)
{
    int retval;
    lm_string_t addr = NULL;

    /* this is global */
    mail_from_rfc_formatted = 0;

    SYSLOG((LOG_INFO, "MAIL command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* Parse the address from the arguments */
    if (smtp_parse_address(args, &addr, &mail_from_rfc_formatted)) {
        SYSLOG((LOG_WARNING, "MAIL syntax error"));
        smtp_printf((outbuf, "555 syntax error (#5.5.4)\r\n"));
        return 0;
    }

    /* Perform reset of internal state */
    smtp_rset();

    /* Set the flag to show we received a mail comand */
    gotmail_flag = 1;

    /* Call the mail callback handler function */
    retval = smtp_mail_callback(addr);
    LM_STRING_FREE(addr);
    if (!retval) {
        /* Print an ok message if callback returned zero */
        smtp_printf((outbuf, "250 ok\r\n"));
    }

    return 0;
}

static int smtp_mail_default_callback(lm_string_t addr)
{
    SYSLOG((LOG_INFO, "MAIL default callback function called, addr : %s",
            LM_STRING_BUFFER(addr)));

    return 0;
};

static int smtp_RCPT_handler(lm_string_t args)
{
    int retval;
    lm_string_t addr = NULL;

    /* this is not currently used outside this function */
    int rcpt_to_rfc_formatted;

    SYSLOG((LOG_INFO, "RCPT command received, args: %s",
            LM_STRING_BUFFER(args)));

    if (gotmail_flag == 0) {
        SYSLOG((LOG_NOTICE, "MAIL first errror"));
        smtp_printf((outbuf, "503 MAIL first (#5.5.1)\r\n"));
        return 0;
    }

    /* Get the address from the arguemnts, printing error message if syntax is incorrect */
    if (smtp_parse_address(args, &addr, &rcpt_to_rfc_formatted)) {
        SYSLOG((LOG_NOTICE, "RCPT syntax error"));
        smtp_printf((outbuf, "555 syntax error (#5.5.4)\r\n"));
        return 0;
    }

    /* Call the rcpt callback handler function */
    retval = smtp_rcpt_callback(addr);

    /* Increment the rcpt flag if the address was taken without an error */
    /* HACK This function now increments the rcpt count when the return code
       is >= 0.  This is required because quarantined mail is still delivered,
       but if the user requests a UDNE return on spam hits the rcpt count has
       to still be delivered and no message printed, where with the old system,
       it would only increment and print the message. */
    if (retval >= 0) {
        gotrcpt_flag++;
    }

    /* Clean up */
    LM_STRING_FREE(addr);

    /* Print an ok message if callback returned zero */
    if (retval == 0) {
        smtp_printf((outbuf, "250 ok\r\n"));
    }

    return 0;
}

/* Don't do anything on default */
static int smtp_rcpt_default_callback(lm_string_t addr)
{
    SYSLOG((LOG_INFO, "RCPT default callback function called, addr : %s",
            LM_STRING_BUFFER(addr)));
    return 0;
}

static void smtp_stray_newline(void)
{
    if (config.stray_newline_detection) {
        SYSLOG((LOG_NOTICE, "Stray newline detected. returning 451"));
        smtp_printf((outbuf,
                     "451 stray newline detected. see http://cr.yp.to/docs/smtplf.html\r\n"));
        smtp_quit_callback(0);
    } else {
        SYSLOG((LOG_NOTICE, "Stray newline detected.  Continuing."));
    }
}

static int smtp_DATA_handler(lm_string_t args)
{
    lm_string_t line = NULL;
    int read_error_flag;
    int endmarker_flag;
    int retval;
    int in_header_flag;

    SYSLOG((LOG_INFO, "DATA command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* If we haven't gotten a mail command print error */
    if (gotmail_flag == 0) {
        SYSLOG((LOG_NOTICE, "MAIL first error "));
        smtp_printf((outbuf, "503 MAIL first (#5.5.1)\r\n"));
        return 0;
    }

    /* If we haven't gotten a valid rcpt command print error */
    if (gotrcpt_flag == 0) {
        SYSLOG((LOG_NOTICE, "RCPT first error "));
        smtp_printf((outbuf, "503 RCPT first (#5.5.1)\r\n"));
        return 0;
    }

    /* Call the queue init function */
    LM_ASSERT(smtp_queue_init_callback);
    smtp_queue_init_callback();

    /* Print message to sender to send data */
    smtp_printf((outbuf, "354 go ahead\r\n"));

    /* Read the header from stdin */
    endmarker_flag = 0;
    in_header_flag = 1;
    read_error_flag = 0;
    while (!read_error_flag && !endmarker_flag) {
        char *buffer;

        read_error_flag = smtp_read_callback(0, &line);

        if (read_error_flag) {
            SYSLOG((LOG_WARNING, "error reading data line!\n"));
            return 0;
        }

        buffer = LM_STRING_BUFFER(line);
        switch (buffer[0]) {
                /* Look for end of header */
            case '\r':
                if (buffer[1] == '\n')
                    in_header_flag = 0;
                break;

            case '.':
                switch (buffer[1]) {
                        /* double period at start of line */
                        /* convert to single period */
                    case '.':
                        LM_STRING_START_SHIFT(line, 1);
                        break;

                        /* ^.\r */
                    case '\r':
                        /* .\r\n end of data */
                        if (buffer[2] == '\n')
                            endmarker_flag = 1;
                        break;

                        /* ^.\n */
                    case '\n':
                        /* stray new line */
                        smtp_stray_newline();
                        break;

                    default:
                        /* do nothing */
                        break;
                }
                break;

            case '\n':
                /* stray new line */
                smtp_stray_newline();
                break;

            default:
                break;
        }

        /* if we didn't get any of the above cases, search
           the string and ensure the line is ended with
           CRLF */
        while (!endmarker_flag && *buffer != '\0') {
            switch (buffer[0]) {
                case '\r':
                    if (buffer[1] == '\n') {
                        /* Remove the \r's from the message (which are replaced 
                           by qmail-remote when the message is sent.  I would assume 
                           that this is the behaviour of all outgoing remote clients 
                           (or I'll at least have to assume this because we are going 
                           for qmail compatibility) */
                        buffer[0] = '\n';
                        LM_ASSERT(buffer[2] == '\0');
                        LM_STRING_END_SHIFT(line, 1);
                        buffer++;
                    }
                    break;

                case '\n':
                    /* stray newline */
                    smtp_stray_newline();
                    break;

                default:
                    /* don't do anything */
                    break;
            }
            buffer++;
        }

        if (in_header_flag) {
            if (!endmarker_flag && smtp_header_line_callback(line)) {
                /* Execute callback for each line in the header  */
                SYSLOG((LOG_ERR, "error with header line callback!"));
            }
        } else {
            /* Execute callback for each body line  */
            if (!endmarker_flag && smtp_body_line_callback(line)) {
                SYSLOG((LOG_ERR, "error with body line callback!"));
            }
        }

        LM_STRING_FREE(line);
    }

    /* Call the queue function */
    if (smtp_queue_callback()) {
        SYSLOG((LOG_ERR, "error queueing message"));
        return -1;
    }

    /* Call the queue cleanup function */
    LM_ASSERT(smtp_queue_cleanup_callback);
    smtp_queue_cleanup_callback();

    retval = smtp_data_callback();

    smtp_rset_callback();

    return 0;
}

static int smtp_data_default_callback(void)
{
    SYSLOG((LOG_INFO, "DATA default callback function called.\n"));

    smtp_printf((outbuf, "250 ok Default callback called - nothing done\r\n"));

    return 0;
}

static int smtp_AUTH_handler(lm_string_t args)
{
    lm_string_t type = NULL;
    int authtype = -1;
    lm_string_t username64 = NULL;
    lm_string_t typestring = NULL;
    int index = 0;

    SYSLOG((LOG_INFO, "AUTH command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* If there is a space in the args, treat the first 
       token as the auth type and the second to be the
       username */
    typestring = args;
    if (!LM_STRING_CHR(args, ' ', index)) {
        if (LM_STRING_DUP(username64, args)
            || LM_STRING_START_SHIFT(username64, index + 1)
            || LM_STRING_CHOP_HEAD(username64)
            || LM_STRING_END_SHIFT(args, LM_STRING_LEN(args) - index)) {
            SYSLOG((LOG_ERR, "error creating base64 username"));
            return -1;
        }
    }

    /* Decode the type */
    if (authtype == -1) {
        if (LM_STRING_NEW(type, SMTP_AUTH_TYPE_STRING_LOGIN)) {
            return -1;
        }
        if (!LM_STRING_CASECMP(type, args)) {
            authtype = SMTP_AUTH_LOGIN;
        }
        LM_STRING_FREE(type);
    }

    if (authtype == -1) {
        if (LM_STRING_NEW(type, SMTP_AUTH_TYPE_STRING_PLAIN)) {
            return -1;
        }
        if (!LM_STRING_CASECMP(type, args)) {
            authtype = SMTP_AUTH_PLAIN;
        }
        LM_STRING_FREE(type);
    }

    if (authtype == -1) {
        if (LM_STRING_NEW(type, SMTP_AUTH_TYPE_STRING_CRAM_MD5)) {
            return -1;
        }
        if (!LM_STRING_CASECMP(type, args)) {
            authtype = SMTP_AUTH_CRAM_MD5;
        }
        LM_STRING_FREE(type);
    }

    if (authtype == -1) {
        if (LM_STRING_NEW(type, SMTP_AUTH_TYPE_STRING_DIGEST_MD5)) {
            return -1;
        }
        if (!LM_STRING_CASECMP(type, args)) {
            authtype = SMTP_AUTH_DIGEST_MD5;
        }
        LM_STRING_FREE(type);
    }

    /* Call the auth callback handler function */
    if (smtp_auth_callback(authtype, username64)) {
        return 0;
    }

    return 0;
}

/* Handle a QUIT command.  Print quit message 221 and exit the program with a
   return value of 0 */
static int smtp_QUIT_handler(lm_string_t args)
{
    int retval;

    SYSLOG((LOG_INFO, "QUIT command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* Free this to keep it out of the memory dump */
    LM_STRING_FREE(args);

    /* Cleanup and quit */
    LM_ASSERT(smtp_quit_callback);
    retval = smtp_quit_callback(0);
    if (retval != 0) {
        return 0;
    }

    smtp_printf((outbuf, "221\r\n"));
    exit(retval);
}

static int smtp_RSET_handler(lm_string_t args)
{
    SYSLOG((LOG_INFO, "RSET command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* Call the rset callback handler function */
    if (smtp_rset_callback()) {
        return 0;
    }

    smtp_printf((outbuf, "250 flushed\r\n"));
    return smtp_rset();
}

static int smtp_HELP_handler(lm_string_t args)
{
    SYSLOG((LOG_INFO, "HELP command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* Call the help callback handler function */
    if (smtp_help_callback()) {
        return 0;
    }

    smtp_printf((outbuf, "214  no help available\r\n"));
    return 0;
}

static int smtp_NOOP_handler(lm_string_t args)
{
    SYSLOG((LOG_INFO, "NOOP command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* Call the noop callback handler function */
    if (smtp_noop_callback()) {
        return 0;
    }

    smtp_printf((outbuf, "250 ok\r\n"));
    return 0;
}

static int smtp_VRFY_handler(lm_string_t args)
{
    SYSLOG((LOG_INFO, "VRFY command received, args: %s",
            LM_STRING_BUFFER(args)));

    /* Call the vrfy callback handler function */
    if (smtp_vrfy_callback(args)) {
        return 0;
    }

    smtp_printf((outbuf, "252 send some mail, i'll try my best\r\n"));
    return 0;

}

static int smtp_rset(void)
{
    gotmail_flag = 0;
    gotrcpt_flag = 0;

    return 0;
}

/* Cleanup memory */
int smtp_cleanup(void)
{
    return 0;
}

/* This function will attempt to match domain with wcdomain, where
wcdomain can be an actual domain or a wildcard domain.  A wildcard domain
is one that has a period as it's first character, and will match any
domains with a prefix before the domain */
int smtp_match_domain(lm_string_t domain, lm_string_t wcdomain)
{
    int match = 1;

    /* Check for a leading dot on the domain, which represents a wildcard */
    if (LM_STRING_BUFFER(wcdomain)[0] == '.') {
        lm_string_t tempdomain = NULL;
        lm_string_t tempwcdomain = NULL;

        /* If the domain is shorter than the wildcard domain without the 
         * leading period, we can't possibly have a match so we won't 
         * bother checking 
         */
        if (LM_STRING_LEN(domain) < LM_STRING_LEN(wcdomain) - 1) {
            return 1;
        }

        /* Make scratch copies of the domains */
        if (LM_STRING_DUP(tempdomain, domain)
            || LM_STRING_DUP(tempwcdomain, wcdomain)) {
            return -1;
        }

        /* Shift off the first dot in the domain */
        LM_STRING_START_SHIFT(tempwcdomain, 1);

        /* While there is a possibility of a match and there isn't a match... */
        while ((LM_STRING_LEN(tempdomain)) >= LM_STRING_LEN(tempwcdomain)
               && (match != 0)) {
            /* If the first character of the strings and lengths match */
            if ((LM_STRING_BUFFER(tempdomain)[0] ==
                 LM_STRING_BUFFER(tempwcdomain)[0])
                && (LM_STRING_LEN(tempdomain) == LM_STRING_LEN(tempdomain))) {
                /* See if the strings match */
                match = LM_STRING_CMP(tempwcdomain, tempdomain);
            }

            /* Shift off the first character in the string and
             * look again */
            LM_STRING_START_SHIFT(tempdomain, 1);
        }

        /* Free the strings */
        LM_STRING_FREE(tempdomain);
        LM_STRING_FREE(tempwcdomain);
    } else {
        /* If the strings are different length, don't bother  checking */
        if (LM_STRING_LEN(domain) != LM_STRING_LEN(wcdomain)) {
            return 1;
        }

        match = LM_STRING_CMP(wcdomain, domain);
    }

    return match;
}
