/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Josh Wilsdon <josh@wizard.ca>
 *
 * CVS Id: $Id: mod_dbfile.c,v 1.19 2003/10/27 20:41:02 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <crypt.h>

#include "config_spam_rule.h"
#include "config_qmail.h"
#include "libgetbdb.h"
#include "common/load_rules.h"
#include "common/spam_rules.h"
#include "common/read_rules.h"
#include "auth.h"
#include "mod_dbfile.h"

#include "common/controls.h"

extern config_t config;
extern config_type_t config_type_info[];

/* global + user spam rule structures */
extern spam_rule_t user_rules[];
extern spam_rule_t global_rules[];

static int get_email_id_char(char *user, char **out);

int msd_do_auth_dbfile(int type, lm_string_t username, lm_string_t password)
{
    user_record_t user_rec;
    char *cryptpw;
    DB *dbp;
    int ret;
    lm_string_t at = NULL, tmpuser = NULL;

    dbp = openDBFile(config.user_info_dbfile);
    if (dbp == NULL) {
        /* TODO: syslog this higher up */
        return (LM_AUTH_ERROR);
    }

    if (strchr(LM_STRING_BUFFER(username), '@') == NULL) {
        /* append default domain if needed */
        syslog(LOG_DEBUG, "Username before default: [%s]\n",
               LM_STRING_BUFFER(username));
        LM_STRING_NEW(at, "@");
        LM_STRING_DUP(tmpuser, username);
        LM_STRING_CAT(tmpuser, at);
        LM_STRING_CAT(tmpuser, config_qmail.defaultdomain);
        syslog(LOG_DEBUG, "Username after default: [%s]\n",
               LM_STRING_BUFFER(tmpuser));
        ret = getUserStruct(dbp, LM_STRING_BUFFER(tmpuser), &user_rec);
        LM_STRING_FREE(at);
        LM_STRING_FREE(tmpuser);
    } else {
        ret = getUserStruct(dbp, LM_STRING_BUFFER(username), &user_rec);
    }

    if (ret == 0) {
        /* got user data sucessfully */
        cryptpw = crypt(LM_STRING_BUFFER(password), user_rec.password);
        if (strncmp(cryptpw, user_rec.password, strlen(cryptpw)) == 0) {
            /* password matches */
            closeDBFile(dbp);
            return (LM_AUTH_OK);
        } else {
            closeDBFile(dbp);
            return (LM_AUTH_FAILED);
        }
    } else if (ret > 0) {
        /* user does not exist */
        closeDBFile(dbp);
        return (LM_AUTH_NO_USER);
    }

    closeDBFile(dbp);

    return (LM_AUTH_ERROR);
}

/* 
  Return:
    0 does exist
    1 does not exist
   -1 error
*/
int msd_check_rcpt_user_dbfile(lm_string_t * paddr)
{
    DB *dbp;
    char *realuser, *user;
    int ret;

    dbp = openDBFile(config.user_info_dbfile);
    if (dbp == NULL) {
        return (-1);
    }

    /* TODO: append default domain if needed */

    user = strdup(LM_STRING_BUFFER(*paddr));
    if (user == NULL) {
        closeDBFile(dbp);
        return (-1);
    }
    ret = getRealUser(dbp, user, &realuser);
    free(user);

    closeDBFile(dbp);

    if (ret == 0) {
        // printf("user exists as [%s]\n", realuser);
        free(realuser);
        // printf("RETURNING\n");
        return (0);
    } else if (ret == 1) {
        syslog(LOG_DEBUG, "user does not exist %s", LM_STRING_BUFFER(*paddr));
        return (1);
    } else {
        syslog(LOG_ERR, "error determining if user exists");
        return (0);
    }

    return (0);
}

int load_global_spam_rules_dbfile(config_spam_rule_t * rules)
{
    int retval;

    retval = load_spam_rules("0");
    if (retval == 0) {
        translate_rules_to_old_style(0, rules);
    }

    return (retval);
}

int load_spam_rules(char *email_id)
{
    DB *dbp;
    char *data, *ptr, *key, *value;
    unsigned int len;
    int ret, eid;

    if (email_id != NULL) {
        // printf("EMAIL_ID: [%s]\n", email_id);
        // fflush(stdout);
        dbp = openDBFile(config.spam_rule_dbfile);
        if (dbp == NULL) {
            return (-1);
        }

        /* getSpamRules() returns 1 if not found and -1 on error */
        ret = getSpamRules(dbp, email_id, &len, &data);
        if (ret != 0) {
            closeDBFile(dbp);
            return (ret);
        }

        eid = atoi(email_id);

        /* spam rules are in data */

        ptr = data;
        while ((ptr - data) < len) {
            key = ptr;
            ptr += strlen(key) + 1;
            if ((ptr - data) < len) {
                value = ptr;
                ptr += strlen(value) + 1;

                setSpamRule(eid, key, value);
                // printf("EMAIL_ID[%d] RULE[%s] = [%s]\n", eid, key, value);
                // fflush(stdout);
            } else {
                syslog(LOG_ERR, "FATAL: Corrupted spam rules");
                break;
            }
        }

        closeDBFile(dbp);

        return (0);
    } else {
        syslog(LOG_DEBUG, "DID NOT FIND EMAIL_ID");
    }

    return (-1);
}


int getEmailIdDBFile(char *user)
{
    char * email_id = NULL;
    int ret, eid;

    ret = get_email_id_char(user, &email_id);
    if (ret == 0) {
        assert(email_id != NULL);
        eid = atoi(email_id);
        free(email_id);
        return (eid);
    }

    return (-1);
}

int get_email_id_char(char *user, char **out)
{
    DB *dbp;
    unsigned int len;
    char *realuser, *tmpuser, *data, *email_id = NULL;
    int ret;

    if (user == NULL) {
        return (-1);
    }

    dbp = openDBFile(config.user_info_dbfile);
    if (dbp == NULL) {
        return (-1);
    }

    tmpuser = strdup(user);
    if (tmpuser == NULL) {
        syslog(LOG_DEBUG, "load_user_spam_rules_real(): out of memory");
        closeDBFile(dbp);
        return (-1);
    }

    /* XXX */
    ret = getRealUser(dbp, tmpuser, &data);

    free(tmpuser);
    if (ret == 0) {
        realuser = strdup(data);
        if (realuser == NULL) {
            closeDBFile(dbp);
            return (-1);
        }
        ret = getSpamEmailID(dbp, realuser, &len, &data);
        free(realuser);
        if (ret == 0) {
            email_id = strdup(data);
            if (email_id == NULL) {
                syslog(LOG_DEBUG, "load_user_spam_rules_real(): out of memory");
                closeDBFile(dbp);
                return (-1);
            }

        } else if (ret == 1) {
            syslog(LOG_DEBUG, "load_user_spam_rules_real(): user email_id not found");
            closeDBFile(dbp);
            return (1);
        }
    }
    closeDBFile(dbp);

    if (email_id != NULL) {
        /* copy email_id pointer to out (needs to be free()'d by caller) */
        memcpy(out, &email_id, sizeof(out));
        return (0);
    }

    syslog(LOG_DEBUG, "No email_id for user");
    return (-1);
}

int load_user_spam_rules_real(char *user, int *outeid)
{
    char *email_id = NULL;
    int eid, ret;

    if (get_email_id_char(user, &email_id) != 0) {
        syslog(LOG_DEBUG, "load_user_spam_rules_real(): User does not have spam checking enabled");
        return (-1);
    }

    assert(email_id != NULL);

    ret = load_spam_rules(email_id);
    eid = atoi(email_id);
    free(email_id);
    if (ret == 0) {
        /* loaded rules sucessfully */

        /* pass back the email_id if the user wanted it */
        if (outeid != NULL) {
            *outeid = eid;
        }
        return (0);
    } else if (ret == 1) {
        return (1);
    }

    return (-1);
}

int load_user_spam_rules(char *user, config_spam_rule_t * rules)
{
    int retval;
    int eid;

    retval = load_user_spam_rules_real(user, &eid);
    if ((retval == 0) || (retval == 1)) {
        /* user has spam rules, or at least exists */
        translate_rules_to_old_style(eid, rules);

        /* We need to report as success even if no spam rules, because
         * magic-smtpd reports an error if it gets back anything other than
         * zero
         */
        return (0);
    }

    return (retval);
}

lm_string_t create_strlist(int rule)
{
    spam_rule_entry_t *value;
    int i;
    lm_string_t tmpstring = NULL, appendme = NULL, newline = NULL;

    LM_STRING_NEW(newline, "\n");
    LM_STRING_NEW(tmpstring, "");

    if (user_rules[rule].entries > 0) {
        // printf("[%s][%d entries]\n", user_rules[rule].key, user_rules[rule].entries); 
        value = user_rules[rule].first;
        i = 0;
        while (value != NULL) {
            if (value->data != NULL) {
                if (i == 0) {
                    LM_STRING_FREE(tmpstring);
                    LM_STRING_NEW(tmpstring, value->data);
                    i++;
                } else {
                    LM_STRING_CAT(tmpstring, newline);
                    LM_STRING_NEW(appendme, value->data);
                    LM_STRING_CAT(tmpstring, appendme);
                    LM_STRING_FREE(appendme);
                }
            }
            value = value->next;
        }
    } else {
        // printf("[%s][%d entries]\n", user_rules[rule].key, user_rules[rule].entries);
    }

    LM_STRING_FREE(newline);

    // printf("STRLIST[%s]\n", LM_STRING_BUFFER(tmpstring));
    // fflush(stdout);

    if (LM_STRING_LEN(tmpstring) == 0) {
        LM_STRING_FREE(tmpstring);
        return (NULL);
    }
    return (tmpstring);
}

/* XXX NOTE: This is a hack, and should be removed as soon as possible.
             magic-smtpd should be modified to use the common/spam_rule.h
             style structure as soon as possible.  Once that is done, this
             function and calls to it can be removed.
 */
int translate_rules_to_old_style(int eid, config_spam_rule_t * rules)
{
    lm_string_t tmpstring = NULL;

    memset(rules, 0, sizeof(*rules));

    /* for booleans we store a char 0/1 for false/true */

    rules->spam_check = isEnabled(eid, SPAM_RULE_SPAM_CHECK);
    rules->smtp_blocking = isEnabled(eid, SPAM_RULE_SMTP_BLOCKING);
    rules->valid_from_domain = isEnabled(eid, SPAM_RULE_VALID_FROM_DOMAIN);
    rules->all_global_rules = isEnabled(eid, SPAM_RULE_ALL_GLOBAL_RULES);
    rules->smtp_check = isEnabled(eid, SPAM_RULE_SMTP_CHECK);
    rules->delivery_check = isEnabled(eid, SPAM_RULE_DELIVERY_CHECK);
    rules->require_full_addr = isEnabled(eid, SPAM_RULE_REQUIRE_FULL_ADDR);
    rules->block_mail_from_self =
        isEnabled(eid, SPAM_RULE_BLOCK_MAIL_FROM_SELF);
    rules->block_ip_in_addr = isEnabled(eid, SPAM_RULE_BLOCK_IP_IN_ADDR);
    rules->require_me_in_dest = isEnabled(eid, SPAM_RULE_REQUIRE_ME_IN_DEST);
    rules->valid_bounce = isEnabled(eid, SPAM_RULE_VALID_BOUNCE);
    rules->require_helo = isEnabled(eid, SPAM_RULE_REQUIRE_HELO);
    rules->valid_helo_domain = isEnabled(eid, SPAM_RULE_VALID_HELO_DOMAIN);
    rules->mail_from_strict_addr_parse =
        isEnabled(eid, SPAM_RULE_MAIL_FROM_STRICT_ADDR_PARSE);
    rules->check_ip_reverse_dns =
        isEnabled(eid, SPAM_RULE_CHECK_IP_REVERSE_DNS);
    rules->use_global_from_whitelist =
        isEnabled(eid, SPAM_RULE_USE_GLOBAL_FROM_WHITELIST);
    rules->use_global_from_blacklist =
        isEnabled(eid, SPAM_RULE_USE_GLOBAL_FROM_BLACKLIST);
    rules->use_global_helo_whitelist =
        isEnabled(eid, SPAM_RULE_USE_GLOBAL_HELO_WHITELIST);
    rules->use_global_helo_blacklist =
        isEnabled(eid, SPAM_RULE_USE_GLOBAL_HELO_BLACKLIST);
    rules->use_global_country_whitelist =
        isEnabled(eid, SPAM_RULE_USE_GLOBAL_COUNTRY_WHITELIST);
    rules->use_global_country_blacklist =
        isEnabled(eid, SPAM_RULE_USE_GLOBAL_COUNTRY_BLACKLIST);
    rules->use_global_ip_whitelist =
        isEnabled(eid, SPAM_RULE_USE_GLOBAL_IP_WHITELIST);
    rules->use_global_ip_blacklist =
        isEnabled(eid, SPAM_RULE_USE_GLOBAL_IP_BLACKLIST);
    rules->virus_check =
        isEnabled(eid, SPAM_RULE_VIRUS_CHECK);

    /* for strings we store an lm_string_t with value */

    tmpstring = create_strlist(SPAM_RULE_FROM_WHITELIST);
    if (tmpstring != NULL) {
        STRLIST_NEW(rules->from_whitelist, tmpstring);
        LM_STRING_FREE(tmpstring);
    }

    tmpstring = create_strlist(SPAM_RULE_FROM_BLACKLIST);
    if (tmpstring != NULL) {
        STRLIST_NEW(rules->from_blacklist, tmpstring);
        LM_STRING_FREE(tmpstring);
    }

    tmpstring = create_strlist(SPAM_RULE_HELO_WHITELIST);
    if (tmpstring != NULL) {
        STRLIST_NEW(rules->helo_whitelist, tmpstring);
        LM_STRING_FREE(tmpstring);
    }

    tmpstring = create_strlist(SPAM_RULE_HELO_BLACKLIST);
    if (tmpstring != NULL) {
        STRLIST_NEW(rules->helo_blacklist, tmpstring);
        LM_STRING_FREE(tmpstring);
    }

    tmpstring = create_strlist(SPAM_RULE_COUNTRY_WHITELIST);
    if (tmpstring != NULL) {
        STRLIST_NEW(rules->country_whitelist, tmpstring);
        LM_STRING_FREE(tmpstring);
    }

    tmpstring = create_strlist(SPAM_RULE_COUNTRY_BLACKLIST);
    if (tmpstring != NULL) {
        STRLIST_NEW(rules->country_blacklist, tmpstring);
        LM_STRING_FREE(tmpstring);
    }

    tmpstring = create_strlist(SPAM_RULE_IP_WHITELIST);
    if (tmpstring != NULL) {
        STRLIST_NEW(rules->ip_whitelist, tmpstring);
        LM_STRING_FREE(tmpstring);
    }

    tmpstring = create_strlist(SPAM_RULE_IP_BLACKLIST);
    if (tmpstring != NULL) {
        STRLIST_NEW(rules->ip_blacklist, tmpstring);
        LM_STRING_FREE(tmpstring);
    }

    return (0);
}
