/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *            Josh Wilsdon <josh@wizard.ca>
 *
 * CVS Id: $Id: magic-smtpd.c,v 1.132 2003/10/29 20:24:29 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */


/* Standard C includes */
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/wait.h>
#include <time.h>
#include <fcntl.h>
#include <ctype.h>
#include <errno.h>
#include <sys/stat.h>
#include <signal.h>
#include <string.h>

/* Linux Magic library includes */
#include <liblm.h>

/* magic-smtpd includes */
#include "magic-smtpd.h"
#include "config_msd.h"
#include "config_qmail.h"
#include "config_spam_rule.h"
#include "smtp.h"
#include "cdb.h"
#include "check_user.h"
#include "spam_check.h"
#include "auth.h"
#include "strlist.h"
#include "utils.h"
#include "smtp-limits.h"
#include "common/stopwatch.h"
#include "common/spam_log.h"
#ifdef USE_LM_DATABASE
#include "write_spam_log.h"
#endif

#ifdef RELEASE_VERSION
#ifdef OPENSOURCE
    /* Opensource Release */
    #include "version.h.OPENSOURCE"
#else
    /* Commercial Release */
    #include "version.h.STABLE"
#endif
#else
    /* Development */
    #include "version.h"
#endif

/* NEW STYLE */
#include "common/spam_rules.h"
#include "common/clean_rules.h"
#include "common/controls.h"
#include "common/load_rules.h"
#include "common/load_controls.h"

spam_rule_t user_rules[] = SPAM_RULES;
spam_rule_t global_rules[] = SPAM_RULES;

/* Main configuration data, see config.h for contents */
// config_msd_t config;
config_t config;
config_type_t config_type_info[] = CONFIG_TYPE_DATA;

/* Spam log structure */
extern spam_log_t spam_log;

/* Global readonly variables */
typedef struct tag_rcpt_data_t {
    lm_string_t addr;
    config_spam_rule_t *spam_rules;
    int spam_hit_flag;
} rcpt_data_t;

rcpt_data_t rcpt_list[100];
int rcpt_list_count;

/* Global flag for whether things were formatted correctly */
int mail_from_rfc_formatted = 0;

/* Module static variables */
static lm_string_t smtp_from_addr;  /* from addrss for message */
static lm_string_t smtp_rcpt_list;  /* rcpt list of addresses for message */
static lm_string_t smtp_cmd;    /* current smtp command  */
static int smtp_message_size;   /* current message size */
static int smtp_message_size_overflow;  /* overflow bit for message size */
static int smtp_error_flag;     /* flag set if an error occures  */
static int spam_hit_flag;       /* flag to say if on of the spam rules tested positive  */
static int penv[2];             /* envelope pipe for qmail queue */
static int pdata[2];            /* data pipe for qmail queue */
static int childpid;            /* pid of qmail-queue process */
static int num_hops;            /* number of hops this message has taken acccoring to Delivered and Recieved headers */
static int bad_mail_from_flag;  /* mail from: address was found in the badmailfrom list */
static int rcpt_bracket_ip_flag;    /* flag to say if we got a [d.d.d.d] address in rcpt */
static int num_invalid_rcpt;    /* number of rcpts which resulted in an invalid user  */
static int authenticated_flag;  /* flag to indicate if the user is authenticated */
static char *data_return_msg;   /* message to be printed at the end of the data callback, which may have error information */
static char *readbuf;           /* input buffer */
static int timeout_flag;        /* flag to indicate if reading input has timed out */
static int gotmail_flag;        /* flag to indicate if we have received a mail command */
static int rcpt_delay;          /* amount of seconds to delay when getting a rcpt */
static int rcpt_count;          /* number of valid user rcpts we have received */
static int whitelist;           /* 0 if not on whitelist 1 if on whitelist */
int fakehelohost;

/* Constant strings which are all allocated at the start */

static lm_string_t T = NULL;    /* constant T for rcpt address list */
static lm_string_t F = NULL;    /* constant F for from address */
static lm_string_t atsign = NULL;   /* constant @ */
static lm_string_t V = NULL;    /* virus flag check character code */
static lm_string_t W = NULL;    /* whitelist flag check character code */
static lm_string_t Q = NULL;    /* quarantine flag character code */
static lm_string_t S = NULL;    /* spam flag check character code */
lm_string_t helohost = NULL;    

/* Handle resettting of all internal variables */
int msd_rset_callback(void)
{
    int retval;

    /* Free the old data strings */
    LM_STRING_FREE(smtp_rcpt_list);
    LM_STRING_FREE(smtp_from_addr);

    /* Reset the strings to empty */
    retval = 0;
    LM_STRING_NEW(smtp_rcpt_list, "");
    retval = (smtp_rcpt_list == NULL);
    LM_STRING_NEW(smtp_from_addr, "");
    retval |= (smtp_from_addr == NULL);

    if (retval != 0) {
        msd_error_out_of_memory();
    }

    /* Reset the message size and overflow flag to zero */
    smtp_message_size = 0;
    smtp_message_size_overflow = 0;

    /* Reset the error flag */
    smtp_error_flag = 0;

    /* Reset data callback return message string */
    data_return_msg = 0;

    /* Reset the bad mail from flag */
    bad_mail_from_flag = 0;

    /* Reset the rcpt count callbacks */
    rcpt_count = 0;

    /* Reset the authenticated flag */
    authenticated_flag = 0;

    /* Reset the gotmail flag */
    gotmail_flag = 0;

    /* Reset the spam hit flag */
    spam_hit_flag = 0;

    /* Reset whitelist flag */
    whitelist = 0;

    return 0;
}

/* Perform cleanup operations and exit */
int msd_quit_callback(int retval)
{
    if (retval == 0) {
        fprintf(stdout, "221 %s \r\n",
                LM_STRING_BUFFER(config_qmail.smtpgreeting));
    }

    /* Do cleanup operations */
    /* config_msd_cleanup(); */
    config_qmail_cleanup();
    config_spam_rule_free(&global_spam_rules);
    smtp_cleanup();

    if (smtp_cmd != NULL) {
        LM_STRING_FREE(smtp_cmd);
    }

    /* Free any data that has been read */
    LM_STRING_FREE(smtp_rcpt_list);
    LM_STRING_FREE(smtp_from_addr);
    LM_STRING_FREE(F);
    LM_STRING_FREE(T);
    LM_STRING_FREE(V);
    LM_STRING_FREE(Q);
    LM_STRING_FREE(W);
    LM_STRING_FREE(S);
    LM_STRING_FREE(atsign);

    LM_FREE(readbuf);

    if (config.log_level >= LOG_DEBUG) {
        LM_MEMSTAT(stderr);
        LM_MEMDUMP(stderr);
    }

    SYSLOG((LOG_NOTICE, "exiting"));
    CLOSELOG;

    fflush(stdout);
    fflush(stderr);

    exit(retval);
    /* prevents compiler warnings */
    return retval;
}

void msd_error_unavailable(void)
{
    fprintf(stdout,
            "421 Service not available, closing transmission channel\r\n");
    fflush(stdout);
    /* Just exit here since this can be called before things are initialized and that
       causes problems.  No real need to call cleanup functions anyways since all memory 
       will be freed on exit */

    SYSLOG((LOG_ALERT, "returning service unavailalble"));
    CLOSELOG;
    exit(1);
    /*  msd_quit_callback(1); */
}

void msd_error_out_of_memory(void)
{
    /* Print no mem error message */
    fprintf(stdout, "421 out of memory (#4.3.0)\r\n");
    fflush(stdout);

    /* Die with exit code 1 */
    SYSLOG((LOG_EMERG, "out of memory"));
    msd_quit_callback(1);
}

/* Exec the qmail-queue and setup write pipes to send the envelope and 
   body data as they are provided by the queue callback */
int msd_queue_init(void)
{
    char *args[] = { NULL, NULL };
    char *buffer = NULL;
    int buffersize;
    int numchars;

    args[0] = config.qmail_queue;

    /* Allocate the read buffer */
    buffersize = config.max_line_length;
    if (LM_MALLOC(buffer, buffersize)) {
        LM_ASSERT(buffer == NULL);
        msd_error_out_of_memory();
    }

    /* Reset the number of hops for new message */
    num_hops = 0;

    if ((pipe(penv) == -1) || (pipe(pdata) == -1)) {
        perror("pipe error\n");
    }

    switch (childpid = fork()) {
        case 0:
            /* Child */

            /* close descriptor 0 and re-open it as the read end of pdata */
            close(0);
            dup(pdata[0]);

            /* close the pdata pipe (we've still got descriptor 0 pointing to it) */
            close(pdata[0]);
            close(pdata[1]);

            /* close descriptor 1 and re-open it as the read end of penv */
            close(1);
            dup(penv[0]);

            /* close the penv pipe (we've still got descriptor 1 pointing to it) */
            close(penv[0]);
            close(penv[1]);

            /* run qmail-queue (which reads from descriptors 0 and 1) */
            /* TODO Is this safe? Might need to reconsider executing a string that is 
             * pulled from the environment 
             */
            if (execvp(config.qmail_queue, args)) {
                perror("execvp");
                return -1;
            }
            /* should never get here */
            perror("unreachable code");
            _exit(1);
            break;

        case -1:
            /* fork() error */
            SYSLOG((LOG_ERR, "qmail-queue fork error"));
            break;
    }

    /* Write the Recieved: line from us for the message */
    numchars = snprintf(buffer, buffersize, "Received: from %s ",
                 LM_STRING_BUFFER(config_qmail.tcpremotehost));
    if ((smtp_error_flag == 0) && (numchars > 0)
        && (write(pdata[1], buffer, numchars) != numchars)) {
        smtp_error_flag |= 1;
    }

    /* If we were given a helo domain different than tcp server, print it */
    if (fakehelohost != 0) {
        numchars = snprintf(buffer, buffersize, "(HELO %s) ",
                     LM_STRING_BUFFER(helohost));
        if ((smtp_error_flag == 0) && (numchars > 0)
            && (write(pdata[1], buffer, numchars) != numchars)) {
            smtp_error_flag |= 1;
        }
    }

    /* Write an ( for the remote host information to be contained in */
    numchars = snprintf(buffer, buffersize, "(");
    if ((smtp_error_flag == 0) && (numchars > 0)
        && (write(pdata[1], buffer, numchars) != numchars)) {
        smtp_error_flag |= 1;
    }

    /* Write the remote info if given by tcpserver */
    if (LM_STRING_BUFFER(config_qmail.tcpremoteinfo)[0] != '\0') {
        numchars = snprintf(buffer, buffersize, "%s@",
                     LM_STRING_BUFFER(config_qmail.tcpremoteinfo));
        if ((smtp_error_flag == 0) && numchars > 0
            && write(pdata[1], buffer, numchars) != numchars) {
            smtp_error_flag |= 1;
        }
    }

    /* Write the remote ip */
    numchars = snprintf(buffer, buffersize, "%s)\n by %s with SMTP; ",
                 LM_STRING_BUFFER(config_qmail.tcpremoteip),
                 LM_STRING_BUFFER(config_qmail.tcplocalip));
    if ((smtp_error_flag == 0) && numchars > 0
        && write(pdata[1], buffer, numchars) != numchars) {
        smtp_error_flag |= 1;
    }

    /* Print out the time in the received line */
    /* TODO Not exact format as qmail
       We do: Thu Oct 24 21:54:29 2002 -0000
       Qmail does: 24 Oct 2002 21:54:29 -0000
       Both are gmt tho, so that's good
     */
    {
        time_t t;
        struct tm *times;

        time(&t);

        /* 
         * Print out the time string according to the RFC and man strftime
         */
        times = gmtime(&t);
        numchars = strftime(buffer, buffersize, "%a, %d %b %Y %H:%M:%S %z\n", times);
        if ((smtp_error_flag == 0) && numchars > 0
            && write(pdata[1], buffer, numchars) != numchars) {
            smtp_error_flag |= 1;
        }
    }

    /* Free the buffer */
    LM_FREE(buffer);
    return 0;
}

/* Write the envelope data to qmail-queue  */
int msd_queue(void)
{
    int index;
    int retval;

    /* Need to close the message body pipe so that we can write to the header
       data pipe */
    close(pdata[1]);

    /* Write the envelope to file descriptor 1 */

    /* Write the from address and it's trailing \0 to the message envelope pipe */
    LM_STRING_ASSERT(smtp_from_addr);
    smtp_error_flag |= (LM_STRING_CHECK(smtp_from_addr) != 0);
    if (smtp_error_flag == 0) {
        retval = write(penv[1], "F", 1);
        smtp_error_flag |= (retval != 1);
    }

    /* Write the from addr to the envelope pipe */
    if (smtp_error_flag == 0) {
        retval = write(penv[1], LM_STRING_BUFFER(smtp_from_addr),
                  LM_STRING_LEN(smtp_from_addr));
        smtp_error_flag |= (retval != LM_STRING_LEN(smtp_from_addr));
    }

    /* Write the terminator character for the from address */
    if (smtp_error_flag == 0) {
        retval = write(penv[1], "\0", 1);
        smtp_error_flag |= (retval != 1);
    }

    /* QMAIL */
    /* If there was an error don't write the rcpt list */
    if (smtp_error_flag) {
        SYSLOG((LOG_ERR, "error writing envelope from section!"));
        return 0;
    }

    /* Find the location of the first T in the rcpt list */
    if (LM_STRING_CHR(smtp_rcpt_list, 'T', index)) {
        /* If there isn't one, there was a serious problem */
        SYSLOG((LOG_ERR, "empty rcpt list!!!"));
        return -1;
    }

    /* index should always be 0, since all rcpt addresses start with a T */
    LM_ASSERT(index == 0);
    while (index == 0) {
        /* Shift off the T */
        LM_ASSERT(LM_STRING_BUFFER(smtp_rcpt_list)[0] == 'T');
        LM_STRING_START_SHIFT(smtp_rcpt_list, 1);

        /* Write a T to the pipe */
        if (smtp_error_flag == 0) {
            retval = write(penv[1], "T", 1);
            smtp_error_flag |= (retval != 1);
        }

        /* Find the next T */
        if (LM_STRING_CHR(smtp_rcpt_list, 'T', index)) {
            /* If there wasn't another T, we are at the last address in the list,
               so we write that and then put the two null terminators to signify the
               end of the envelope */
            if (smtp_error_flag == 0) {
                retval =
                    write(penv[1], LM_STRING_BUFFER(smtp_rcpt_list),
                          LM_STRING_LEN(smtp_rcpt_list));
                smtp_error_flag |= (retval != LM_STRING_LEN(smtp_rcpt_list));
            }

            /* this is actually 3 bytes long */
            if (smtp_error_flag == 0) {
                retval = write(penv[1], "\0\0", 2);
                smtp_error_flag |= (retval != 2);
            }
            break;
        } else {
            /* We found another T, so we write the address up to where the T was found 
               and then write a single null terminator and then shift that address off
               the start of the string  */
            LM_ASSERT(index <= LM_STRING_LEN(smtp_rcpt_list));
            if (smtp_error_flag == 0) {
                retval =
                    write(penv[1], LM_STRING_BUFFER(smtp_rcpt_list), index);
                smtp_error_flag |= (retval != index);
            }

            /* this is actually 2 bytes long */
            if (smtp_error_flag == 0) {
                retval = write(penv[1], "\0", 1);
                smtp_error_flag |= (retval != 1);
            }

            LM_STRING_START_SHIFT(smtp_rcpt_list, index);
            index = 0;
        }
    }

    return 0;
}

/* Close the pipes to the qmail-queue process that are still open, and wait for the 
 * child to exit and get it's exit code */
int msd_queue_cleanup(void)
{
    int wpret;
    char childret;
    int retval;

    /* Close the pipe */
    close(penv[1]);

    /* Wait for the child to stop */
    do {
        wpret = waitpid(childpid, &retval, 0);
    } while (wpret == -1 && errno == EINTR);

    /* Check if there was some sort of wierd problem with waitpid */
    if (wpret != childpid) {
        /* TODO not tested */
        data_return_msg = "451 qq waitpid surprise (#4.3.0)\r\n";
        return 0;
    }

    /* Check to see if qmail-queue crashed */
    if (retval & 127) {
        /* TODO not tested */
        data_return_msg = "451 qq crashed (#4.3.0)\r\n";
    }

    /* Check the exti status of qmail queue and print the appropriate return message */
    /* I have bad feelings about WEXITSTATUS returning 0-255. 
     * This will make sure the value is sign extended to an int
     */
    childret = WEXITSTATUS(retval);
    retval = childret;
    if (retval > 0) {
        switch (retval) {
            case 115:
            case 11:
                data_return_msg =
                    "554 envelope address too long for qq (#5.1.3)";
                break;
            case 31:
                data_return_msg =
                    "554 mail server permanently rejected message (#5.3.0)";
                break;
            case 51:
                data_return_msg = "451 qq out of memory (#4.3.0)";
                break;
            case 52:
                data_return_msg = "451 qq timeout (#4.3.0)";
                break;
            case 53:
                data_return_msg = "451 qq write error or disk full (#4.3.0)";
                break;
            case 0:
                if (!smtp_error_flag) {
                    break;
                }
                /* fall through if there was an error */
            case 54:
                data_return_msg = "451 qq read error (#4.3.0)";
                break;
            case 55:
                data_return_msg =
                    "451 qq unable to read configuration (#4.3.0)";
                break;
            case 56:
                data_return_msg =
                    "451 qq trouble making network connection (#4.3.0)";
                break;
            case 61:
                data_return_msg = "451 qq trouble in home directory (#4.3.0)";
                break;
            case 63:
            case 64:
            case 65:
            case 66:
            case 62:
                data_return_msg =
                    "451 qq trouble creating files in queue (#4.3.0)";
                break;
            case 71:
                data_return_msg =
                    "451 mail server temporarily rejected message (#4.3.0)";
                break;
            case 72:
                data_return_msg =
                    "451 connection to mail server timed out (#4.4.1)";
                break;
            case 73:
                data_return_msg =
                    "451 connection to mail server rejected (#4.4.1)";
                break;
            case 74:
                data_return_msg =
                    "451 communication with mail server failed (#4.4.2)";
                break;
            case 91:           /* fall through */
            case 81:
                data_return_msg = "451 qq internal bug (#4.3.0)";
                break;
            case 120:
                data_return_msg = "451 unable to exec qq (#4.3.0)";
                break;
            default:
                if ((retval >= 11) && (retval <= 40)) {
                    data_return_msg = "554 qq permanent problem (#5.3.0)";
                    break;
                }
                data_return_msg = "451 qq temporary problem (#4.3.0)";
                break;
        }
    }

    return retval > 0;
}

/* Check if this message has been bounced or relayed and increment the hop
   count if it has */
void msd_header_check_hop(char *header_line)
{
    char received[] = "Received";
    char delivered[] = "Delivered";

    /*
     * Look for any "Received:" or "Delivered:" headers.  For each of these
     * we'll increment the hop count so that we can detect loops.
     *
     */
    if ((strncasecmp(received, header_line, strlen(received)) == 0)
        || (strncasecmp(delivered, header_line, strlen(delivered)) == 0)) {
        SYSLOG((LOG_NOTICE, "incrementing hop count"));
        num_hops++;
    }

    /* If the number of hops was too great print warning */
    if ((num_hops > config.max_hops) && (config.max_hops > 0)) {
        SYSLOG((LOG_WARNING, "max hop count exceeded!"));

        /* Flag an error for to be caught later */
        smtp_error_flag |= 1;

        fprintf(stdout,
                "554 too many hops, this message is looping (#5.4.6)\r\n");
        fflush(stdout);
        msd_quit_callback(1);
    }
}

int msd_update_msg_size(int size)
{
    unsigned int tmpsize;

    /* Add the length of the line to the total message size */
    /* Check for overflow of smtp_message_size */
    tmpsize = smtp_message_size;
    tmpsize += size;

    /* If the value of tmpsize is less than the old message size,
     * there was a size overflow so we'll flag an error
     */
    if ((unsigned int) tmpsize > (unsigned int) smtp_message_size) {
        smtp_message_size = tmpsize;
        if (config_qmail.databytes != 0) {
            if (smtp_message_size > config_qmail.databytes) {
                smtp_error_flag |= 1;
                return -1;
            }
        }
    } else {
        /* Overflow in size variable error */
        smtp_message_size_overflow = 1;
        smtp_error_flag |= 1;
    }

    return 0;

}

/* This function is called for each line contained in the header
   Simply writes it to the qmail-queue message pipe for now */
int msd_header_line_callback(lm_string_t line)
{
    int retval = 0;

    /* Update the total number of bytes that have been received */
    msd_update_msg_size(LM_STRING_LEN(line));

    /* Perform hop checking like qmail does */
    msd_header_check_hop(LM_STRING_BUFFER(line));

    /* If there hasn't been an error yet, write the line to qmail-queue */
    if (smtp_error_flag == 0) {
        retval = write(pdata[1], LM_STRING_BUFFER(line), LM_STRING_LEN(line));
    } else {
        /* There was an error */

    }

    /* Flag an error if line wasn't written fully */
    smtp_error_flag |= (retval != LM_STRING_LEN(line));

    return 0;
}

/* This function is called for each line contained in the header
   Simply writes it to the qmail-queue message pipe for now */
int msd_body_line_callback(lm_string_t line)
{
    int retval = 0;

    /* Update the total number of bytes that have been received */
    msd_update_msg_size(LM_STRING_LEN(line));

    /* Write the line to the data pipe */
    if (!smtp_error_flag) {
        retval = write(pdata[1], LM_STRING_BUFFER(line), LM_STRING_LEN(line));
    }

    /* Flag an error if line wasn't written fully */
    smtp_error_flag |= (retval != LM_STRING_LEN(line));

    return 0;
}

/* Perform authorization with the given username and password, returning
   0 on sucess or
   * -1 on failure and setting the authorized_flag to true */
int msd_auth_callback(int authtype, lm_string_t username64)
{
    int retval = 0;

    /* If auth is not enabled, print error */
    if (config.auth_enable == 0) {
        fprintf(stdout, "503 auth not available (#5.3.3)\r\n");
        fflush(stdout);
        return -1;
    }

    /* If we've already gotten a mail command, can't do an auth */
    if (gotmail_flag != 0) {
        fprintf(stdout, "503 no auth during mail transaction (#5.5.0)\r\n");
        fflush(stdout);
        return -1;
    }

    /* If already authorized, print error */
    if (authenticated_flag != 0) {
        fprintf(stdout, "503 you're already authenticated (#5.5.0)\r\n");
        fflush(stdout);
        return -1;
    }

    /* Do authorization */
    retval = msd_do_auth(authtype, username64);

    /* Check the return code and set retval accordingly */
    if (retval == 0) {
        /* Print authorization denedied message and set retval to -1 */
        authenticated_flag = 1;
        setenv("RELAYCLIENT", "", 0);
        fprintf(stdout, "235 ok, go ahead (#2.0.0)\r\n");
    } else if (retval > 0) {
        /* Authorization denied, print error */
        fprintf(stdout, "535 authorization failed (#5.7.0)\r\n");
        SYSLOG((LOG_NOTICE, "Sleeping auth failure"));
        sleep(5);
    } else {
        switch (retval) {
            case LM_AUTH_UNSUPPORTED:
                fprintf(stdout, "504 Unsupported authorization type.\r\n");
                break;
            case LM_AUTH_ERROR_DB:
                fprintf(stdout,
                        "454 Temporary authentication failure.\r\n");
                break;
            case LM_AUTH_CANCEL:
                fprintf(stdout, "501 Authorization canceled\r\n");
                break;
            case LM_AUTH_BAD_DATA:
                fprintf(stdout,
                        "501 Username or password cannot be decoded.  Authorization canceled.\r\n");
                break;
            case LM_AUTH_ERROR:
            default:
                fprintf(stdout,
                        "454 problems performing authorization (#4.3.0)\r\n");
                break;
        }
        SYSLOG((LOG_NOTICE, "Sleeping after auth problem/failure"));
        sleep(5);
    }
    fflush(stdout);

    /* Return non-zero to prevent further processing by caller */
    return 1;
}

/* Function called after all queuing functions have completed */
int msd_data_callback(void)
{
    /* Check if the message size is greater than maximum size and print 
     * error return code if
     * Also check if there was an overflow.
     */
    if ((smtp_message_size > config_qmail.databytes)
        || smtp_message_size_overflow) {
        SYSLOG((LOG_WARNING, "message size greater than databytes!"));
        fprintf(stdout,
                "552 sorry, that message size exceeds my databytes limit (#5.3.4)\r\n");
        fflush(stdout);
        return 0;
    }

    /* Print the current time and the pid of the child process for return
       code message string */
    if (data_return_msg) {
        SYSLOG((LOG_DEBUG, "Data returning: %s", data_return_msg));
        fprintf(stdout, "%s\r\n", data_return_msg);
    } else {
        SYSLOG((LOG_DEBUG, "Returning 250 ok for data"));
        fprintf(stdout, "250 ok %d qp %d\r\n", (int) time(NULL), childpid);
    }
    fflush(stdout);

    return 0;
}

/* Check to see if the given address is in the bad mail from list */
int msd_bad_mail_from(lm_string_t addr)
{
    return !strlist_search(config_qmail.badmailfromlist, addr,
                           strlist_atdomain_cmp);
}

/* Function called after all queuing functions have completed */
int msd_mail_callback(lm_string_t addr)
{
    /* Perform a reset when this is called */
    msd_rset_callback();

    /* TODO parse out the address */

    /* Check the address to see if it's in the badmailfrom list */
    if (msd_bad_mail_from(addr) != 0) {
        bad_mail_from_flag = 1;
        return 0;
    }

    /* Set the gotmail flag for the authentication part */
    gotmail_flag = 1;

    /* Print an ok message if everything was ok */
    fprintf(stdout, "250 ok\r\n");
    fflush(stdout);

    /* Save the parsed address */
    LM_STRING_FREE(smtp_from_addr);
    if (LM_STRING_DUP(smtp_from_addr, addr) != 0) {
        msd_error_out_of_memory();
    }

    return 1;
}

/* Check the given hostname to see if it is contained in the rcpthosts 
 * file and we should accept the mail 
 */
int msd_rcpt_host(lm_string_t * paddr)
{
    lm_string_t domain = NULL;
    int retval = 0;
    lm_string_t addr;

    LM_STRING_ASSERT(*paddr);
    addr = *paddr;

    /* Reset the bracket IP flag for each RCPT */
    rcpt_bracket_ip_flag = 0;

    /* Get the domain part of the address */
    if (get_domain(&domain, addr)) {
        msd_error_out_of_memory();
    }

    /* If the address doesn't have a domain
       no need to check rcpt hosts */
    if (LM_STRING_LEN(domain) == 0) {
        LM_STRING_FREE(domain);
        return 0;
    }

    /* Check for local ip address after @ sign that matches
     * machine ip, which would also be local delivery */
    /* THIS IS BROKEN AND ASSUMES ALL @[...] ADDRESSES ARE
       TO THE LOCAL MACHINE!!!! */
    if (LM_STRING_BUFFER(domain)[0] == '[') {
        SYSLOG((LOG_DEBUG, "Recieved mail with [ip] domain part: %s",
                LM_STRING_BUFFER(addr)));

        /* HACK Set the bracket IP flag so that we don't 
           do any valid user or spam checking later */
        rcpt_bracket_ip_flag = 1;

        if (LM_STRING_BUFFER(domain)[LM_STRING_LEN(domain) - 1] != ']') {
            /* no close bracket */
            LM_STRING_FREE(domain);
            return -1;
        }

        LM_STRING_START_SHIFT(domain, 1);
        LM_STRING_END_SHIFT(domain, 1);

        /* NOTE: We only use the IP given in localiphosts instead
           of building the interface list list qmail-smtpd */

        /* Check to see if the IP matches our localiphost */
        if (!LM_STRING_CMP(domain, config_qmail.localiphost)) {
            lm_string_t user = NULL;

            /* Build the new email address with the value of 
               localiphost as the domain */
            get_user(&user, addr);
            LM_STRING_FREE(addr);
            if (LM_STRING_CAT(user, atsign)
                || LM_STRING_CAT(user, config_qmail.localiphost)
                || LM_STRING_DUP(addr, user)) {
                msd_error_out_of_memory();
            }

            SYSLOG((LOG_DEBUG, "IP matched localiphost.  Delivering to: %s",
                    LM_STRING_BUFFER(addr)));

            *paddr = addr;
            LM_STRING_FREE(user);
            retval = 0;
        } else {
            /* The IP didn't match.  We should check all interfaces
               for this machine but don't (yet) */
            SYSLOG((LOG_DEBUG, "IP didn't match localiphost"));

            /* Check if relayclient is set, and if it is, just pass
               the mail through to qmail, else don't accept */
            if (config_qmail.tcprelayclient != NULL) {
                /* Relayclient set, return ok */
                SYSLOG((LOG_DEBUG, "relayclient set, accepting mail for host"));
                retval = 0;
            } else {
                /* Relayclient not set, don't deliver */
                SYSLOG((LOG_DEBUG,
                        "relayclient not set, rejecting delivery for host: %s",
                        LM_STRING_BUFFER(domain)));
                retval = -1;
            }
        }
        if (domain) {
            LM_STRING_FREE(domain);
        }
        return retval;
    }

    retval =
        strlist_search(config_qmail.rcpthostslist, domain,
                       strlist_dotdomain_cmp);
    LM_STRING_FREE(domain);
    return retval;
}


/* XXX Oct 22, 2003: Josh
 *
 *  This function does way too much.  Needs to be split up.
 *
 */
int msd_rcpt_callback(lm_string_t rcptaddr)
{
    int retval = 0;
    int spam_check_retval = 0;
    lm_string_t spamdir = NULL;
    lm_string_t addr = NULL;
    lm_string_t check_flag_sep = NULL;
    config_spam_rule_t spam_rules;
    int in_rcpt_hosts;
    int spam_rules_loaded_flag = 0;
    lm_string_t user, domain;
    stopwatch_t sw;

    SYSLOG((LOG_DEBUG, "RCPT: %s\n", LM_STRING_BUFFER(rcptaddr)));

    /* Increment the number of rcpts that we've received */
    rcpt_count++;

    /* Check how many rcpt's we've gotten and do a delay if over given value */
    if ((config.rcpt_delay_at > 0) && (rcpt_count > config.rcpt_delay_at)) {
        if (rcpt_delay < config.rcpt_delay_max) {
            rcpt_delay += config.rcpt_delay_inc;
        }
        SYSLOG((LOG_NOTICE, "Sleeping for %d seconds for RCPT delay",
                rcpt_delay));
        sleep(rcpt_delay);
    }

    /* Check how many rcpt to's we've gotten and return error if 
     *    over the max value given */
    if ((config.max_rcpt > 0) && (rcpt_count > config.max_rcpt)) {
        SYSLOG((LOG_NOTICE, "Rejecting RCPT.  Too many recipients"));

        /* print temp error message and continue */
        fprintf(stdout, "452 Too many recipients (#4.5.3.1)\r\n");
        fflush(stdout);

        /* Prevent an overflow from a very determined sender */
        rcpt_count = config.max_rcpt;
        return -1;
    }

    /* If the sender was in the bad mail from list, print an error mesage and return */
    if (bad_mail_from_flag) {
        SYSLOG((LOG_NOTICE, "Rejecting RCPT.  In badmailfrom list."));
        fprintf(stdout, "553 sorry, your envelope sender is in my badmailfrom list (#5.7.1)\r\n");
        fflush(stdout);
        return -1;
    }

    /* Make a copy of the address so we can play with it */
    LM_STRING_DUP(addr, rcptaddr);

    /* check to see if we are in the rcpt hosts */
    in_rcpt_hosts = (msd_rcpt_host(&addr) == 0);
    if (in_rcpt_hosts) {
        retval = 0;

        /*  Check to see if the rcpt address is a valid user */
        if ((!rcpt_bracket_ip_flag) && (config.check_valid_users != 0)) {
            retval = msd_check_rcpt_user(&addr, &spamdir);
        } else {
            syslog(LOG_NOTICE, "Domain not in locals, [] address receved or valid user checking disabled");
        }

        /* If the user wasn't found in the database */
        if (retval > 0) {
            /* Increment the number of invalid user RCPTS */
            num_invalid_rcpt++;

            if ((config.max_invalid_rcpt <= num_invalid_rcpt) && (config.max_invalid_rcpt > 0)) {
                fprintf(stdout, "550 Too many invalid user RCPT's sent. Exiting.\r\n");
                syslog(LOG_NOTICE, "too many invalid rcpts: %d", num_invalid_rcpt);
                msd_quit_callback(1);
            }

            /* Output user does not exist */
            fprintf(stdout, "550 User does not exist\r\n");
            fflush(stdout);

            /* Log message about denying mail */
            SYSLOG((LOG_NOTICE, "%s BLOCKED as non-valid user",
                    LM_STRING_BUFFER(addr)));

            /* Return error so that rcpt counter is not
               incremented */
            if (spamdir != NULL) {
                LM_STRING_FREE(spamdir);
            }

            LM_STRING_FREE(addr);
            return -1;
        } else if (retval < 0) {
            /* There was some error with checking the user, 
               but still pass the mail through to qmail 
             */
            SYSLOG((LOG_ERR, "Error checking for valid user: %s",
                    LM_STRING_BUFFER(addr)));
        }
        /* User was valid */
    } else {
        /* If the relayclient variable was set by tcp server */
        if (config_qmail.tcprelayclient != NULL) {
            SYSLOG((LOG_NOTICE, "Relayclient set. Appending to address"));

            /* Append the value of relayclient after the address */
            if (LM_STRING_CAT(addr, config_qmail.tcprelayclient)) {
                msd_error_out_of_memory();
            }
        } else {
            /* Return error so that the caller knows that this wasn't a good rcpt */
            SYSLOG((LOG_ERR, "Rejecting RCPT. Domain not in rcpthosts."));
            fprintf(stdout,
                    "553 sorry, that domain isn't in my list of allowed rcpthosts (#5.7.1)\r\n");
            fflush(stdout);
            LM_STRING_FREE(addr);
            return -1;
        }
    }

    /* We know that the user was valid.  If spamdir has a non-null value,
       the user has a file which matches the address and want's spam checking
       on that address.  We should load the spam configuration variables now */
    /* Added rcpt_bracket_ip_flag so that we don't do any spam checking if
       we received an address list user@[1.2.3.4], which always has to be accepted
       if the given ip matches one of ours */

    if (in_rcpt_hosts && !rcpt_bracket_ip_flag) {

        /* clean out any existing rules */
        cleanRules(user_rules);

        /* Load the spam rules */

        stopwatch(&sw, STOPWATCH_START);
        retval = config_spam_rule_load(addr, &spam_rules, spamdir);

        /* if (retval == 0) {
            printf("[user]\n");
            config_spam_rule_print(spam_rules);
            printf("[global]\n");
            config_spam_rule_print(global_spam_rules);
        } */

        syslog(LOG_DEBUG, "config_spam_rule_load() took %f seconds\n",
               stopwatch(&sw, STOPWATCH_STOP));

        /* Special case of if relayclient is set we wont do the
           helo domain resolves check */
        if (config_qmail.tcprelayclient != NULL) {
            syslog(LOG_NOTICE, "relayclient set.  Disabling valid helo domain spam rule.");
            spam_rules.valid_helo_domain = 0;
        }

        if (retval == 0) {
            /* Spam rules loaded ok */
            syslog(LOG_INFO, "User spam rules loaded successfully");

            spam_rules_loaded_flag = 1;

            /* Check to see if the user has spam checking enabled */
            if (spam_rules.spam_check != 0) {
                syslog(LOG_INFO, "User spam checking enabled");
                user = domain = NULL;

                if (get_user_domain(&user, &domain, addr) == 0) {
                    initSpamLog(LM_STRING_BUFFER(user),
                                LM_STRING_BUFFER(domain),
                                LM_STRING_BUFFER(smtp_from_addr));
                } else {
                    initSpamLog(NULL, NULL, LM_STRING_BUFFER(smtp_from_addr));
                }

                spam_check_retval = spam_check_do(&spam_rules, addr, 
                                                  smtp_from_addr, &whitelist);
                if (whitelist == 1) {
                    SYSLOG((LOG_INFO, "Found on Whitelist"));
                }

#ifdef USE_LM_DATABASE
                if ((config.use_database != 0) && (config.spam_log_db != 0)) {
                    writeSpamLog(config.spam_log_db, NULL);
                }
#endif

                /* XXX: MEMORY MAY BE LOST HERE */
            } else {
                SYSLOG((LOG_INFO, "User does not have spam checking enabled"));
            }
        } else {
            /* There was an error loading the spam rules */
            syslog(LOG_ERR, "Error loading spam rules for '%s'\n",
                   LM_STRING_BUFFER(addr));
            spam_check_retval = -1; /* Signify error */
        }
    } else {
        SYSLOG((LOG_INFO,
                "Domain not in locals, [] address receved or spam checking disabled"));
            syslog(LOG_INFO, "flag: %d valid_users: %d", rcpt_bracket_ip_flag, config.check_valid_users);
    }

    /* Free the spamdir since we don't need it anymore */
    if (spamdir != NULL) {
        LM_STRING_FREE(spamdir);
    }

    /* Set the global spam hit variable for the current message */
    spam_hit_flag = spam_check_retval > 0;

    /* add on a quarantine extention to the address if we have a spam
       message and blocking wasn't requested.
       add on virus and spam checking flags to the end of the address
       to tell the delivery agent to perform them */
    {
        int tmpret = 0;
        lm_string_t tmpdomain = NULL;
        lm_string_t tmpaddr = NULL;

        /* seperate the email address */
        tmpret |= get_user_domain(&tmpaddr, &tmpdomain, addr);

        /* add the check flag prefix we haven't had a spam hit and spam checking
           is enabled or if virus checking is enabled, also ensure address is local */
        if (!tmpret && config.add_check_flags && in_rcpt_hosts) {
            /* if we didn't have a spam hit and we have spam checking enabled OR
               virus checking is enabled add the flag seperator string to the box */

            tmpret |= LM_STRING_NEW(check_flag_sep, "-%-");

            if (!tmpret && ((!spam_hit_flag && spam_rules.spam_check)
                            || (spam_hit_flag && !spam_rules.smtp_blocking)
                            || (spam_rules.virus_check) || whitelist)) {
                tmpret |= LM_STRING_CAT(tmpaddr, check_flag_sep);
            }

            /* add the whitelist flag if message matched whitelist, and do not 
             * add the spam_check flag (possibly still add the virus flag) */
            if (!tmpret && whitelist && spam_rules.spam_check) {
                SYSLOG((LOG_INFO, "Adding flag for whitlelisted entry"));
                tmpret |= LM_STRING_CAT(tmpaddr, W);
            }

            /* add the quarantine extention we had a spam hit and blocking is off */
            if (!tmpret && spam_hit_flag && !whitelist
                && !spam_rules.smtp_blocking) {
                tmpret |= LM_STRING_CAT(tmpaddr, Q);
            }

            /* add the spam check flag if no spam hit and spam checking is enabled and
             * not whitelisted */
            if (!tmpret
                && (!spam_hit_flag && !whitelist && spam_rules.spam_check)) {
                SYSLOG((LOG_INFO, "Adding flags for delivery spam checking"));
                tmpret |= LM_STRING_CAT(tmpaddr, S);
            }

            /* add the virus check flag if virus checking is enabled */
            if (!tmpret && spam_rules.virus_check) {
                SYSLOG((LOG_INFO, "Adding flags for delivery virus checking"));
                tmpret |= LM_STRING_CAT(tmpaddr, V);
            }
        }

        /* unset this since we're done with it */
        whitelist = 0;

        /* rebuild the full address */
        if (!tmpret && (LM_STRING_CAT(tmpaddr, atsign)
                        || LM_STRING_CAT(tmpaddr, tmpdomain))) {
            msd_error_out_of_memory();
        }

        if (addr) {
            LM_STRING_FREE(addr);
        }

        tmpret |= LM_STRING_DUP(addr, tmpaddr);
        if (tmpret) {
            msd_error_out_of_memory();
        }

        if (tmpdomain) {
            LM_STRING_FREE(tmpdomain);
        }

        if (tmpaddr) {
            LM_STRING_FREE(tmpaddr);
        }
    }

    /* if we weren't a spam message or there was an error checking for spam OR
       we were spam but we aren't doing blocking */
    if (spam_check_retval <= 0
        || (spam_check_retval > 0 && !spam_rules.smtp_blocking)) {
        /* if we had a spam hit but blocking was off we are quarantining the message 
           so print a log message saying so */
        if (spam_hit_flag && !spam_rules.smtp_blocking) {
            SYSLOG((LOG_NOTICE,
                    "QUARANTINE: Any mail data received will be sent to %s.",
                    LM_STRING_BUFFER(addr)));
        }

        /* Add the rcpt to the rcpt list */
        if (LM_STRING_CAT(smtp_rcpt_list, T)
            || LM_STRING_CAT(smtp_rcpt_list, addr)) {
            msd_error_out_of_memory();
        }
        retval = 0;
    } else {
        SYSLOG((LOG_NOTICE,
                "BLOCK: Any mail data received will not be delivered."));

        /* don't add to the rcpt list and give error */
        fprintf(stdout, "550 User does not exist\r\n");
        fflush(stdout);

        /* Don't print default 250 return message */
        retval = -1;
    }

    if (spam_rules_loaded_flag) {
        config_spam_rule_free(&spam_rules);
    }

    if (addr) {
        LM_STRING_FREE(addr);
    }

    return retval;
}

int msd_helo_callback(lm_string_t domain)
{
    /* Free the old helo host string and copy the arguents into a new one */
    LM_STRING_FREE(helohost);
    if (LM_STRING_DUP(helohost, domain) != 0) {
        msd_error_out_of_memory();
    }

    /* Set the fakehelohost variable to show that the given helo host doesn't match
       the one provided by tcp server */
    /* Returns 0 on match, fakehelohost set to non zero if they don't */
    fakehelohost =
        LM_STRING_CASECMP(config_qmail.tcpremotehost, helohost);

    fprintf(stdout, "250 %s \r\n", LM_STRING_BUFFER(config_qmail.smtpgreeting));
    fflush(stdout);

    /* Return non-zero to stop default message */
    return -1;
}

int msd_ehlo_callback(lm_string_t domain)
{
    /* Free the old helo host string and copy the arguents into a new one */
    LM_STRING_FREE(helohost);
    if (LM_STRING_DUP(helohost, domain) != 0) {
        msd_error_out_of_memory();
    }

    /* Set the fakehelohost variable to show that the given helo host doesn't match
       the one provided by tcp server */
    /* Returns 0 on match, fakehelohost set to non zero if they don't */
    fakehelohost =
        LM_STRING_CASECMP(config_qmail.tcpremotehost, helohost);

    fprintf(stdout, "250-%s\r\n", LM_STRING_BUFFER(config_qmail.smtpgreeting));
    if (config.auth_enable != 0) {
        fprintf(stdout, "250-AUTH LOGIN\r\n");
        fprintf(stdout, "250-AUTH=LOGIN\r\n");
    }
    fprintf(stdout, "250-PIPELINING\r\n");
    fprintf(stdout, "250 8BITMIME\r\n");
    fflush(stdout);

    return 0;
}

/* Print a specific help string for this program */
int msd_help_callback(void)
{
    fprintf(stdout,
            "214  qmail home page: http://pobox.com/~djb/qmail.html, LinuxMagic Support http://www.linuxmagic.com\r\n");
    fflush(stdout);
    return 1;
}

int msd_init(int argc, char **argv)
{
    int retval = 0;

    /* Perform setup by reading config */
    if (config_qmail_load(config.qmailcontroldir)) {
        SYSLOG((LOG_ERR, "configuration loading error, EXITING"));
        CLOSELOG;
        return -1;
    }

    SETLOGMASK(config.log_level);

    /* default the helo string to "unknown" since this is what we compare
     * it to when we check whether it has been changed */
    LM_STRING_NEW(helohost, "unknown");

    fakehelohost = 0;    /* Real helo host given  */

    /* Put a log message in there */
    SYSLOG((LOG_NOTICE, "Linux Magic SMTPD started: connection from %s",
            LM_STRING_BUFFER(config_qmail.tcpremoteip)));

    /* Allocate empty strings for data that can be received */
    if (LM_STRING_NEW(smtp_rcpt_list, "") || LM_STRING_NEW(smtp_from_addr, "")
        || LM_STRING_NEW(F, "F") || LM_STRING_NEW(T, "T")
        || LM_STRING_NEW(V, "v") || LM_STRING_NEW(Q, "q")
        || LM_STRING_NEW(S, "s") || LM_STRING_NEW(W, "w")
        || LM_STRING_NEW(atsign, "@")
        || LM_MALLOC(readbuf, config.max_line_length)) {
        msd_error_out_of_memory();
    }

    /* If anything failed in this we would have exited */
    return retval;
}

/* Just in case segfault handler so we know something happened */
void msd_sigsegfault(int arg)
{
    fprintf(stdout, "550 internal server error (segfault)\r\n");
    fflush(stdout);
    SYSLOG((LOG_ERR, "internal server error, segfault"));
    CLOSELOG;
    exit(1);
}

int main(int argc, char **argv)
{
    int retval;
    int numcmd = 0;
    char compile_date[] = __DATE__;
    char compile_time[] = __TIME__;
    char features[1024];

    features[0] = '\0';

    /* this module cannot be disabled at this time */
    strcat(features, " +EXTPROG");

#ifdef USE_LM_DATABASE
    strcat(features, " +DATABASE");
#endif

#ifdef USE_LM_DBFILE
    strcat(features, " +DBFILE");
#endif

    if (argc == 2) {
        if (strncmp(argv[1], "-v", 2) == 0) {
            if (compile_date[4] == ' ') {
                compile_date[4] = '0';
            }                
            printf("Version: %s (compiled: %s %s)%s\n", MMVERSION_STRING,
                   compile_date, compile_time, features);
            exit(0);
        } else if (strncmp(argv[1], "-s", 2) == 0) {
            loadControls(10);
            exit(0);
        }
    }

    openlog(MSD_LOG_SYS, LOG_PID, LOG_MAIL);
    setlogmask(LOG_UPTO(LOG_ERR));

    /* Disable pipe signals */
    signal(SIGPIPE, SIG_IGN);

    /* setup a segfault handler so we might get a message printed if we crash */
    signal(SIGSEGV, msd_sigsegfault);

    if (loadControls(0) != 0) {
        printf("load controls failed\n");
        msd_error_unavailable();
    }

    /* Initalize */
    if ((retval = msd_init(argc, argv)) != 0) {
        /* Little hack to keep the 221 quit message from being printed */
        syslog(LOG_ERR, "error initializing");
        msd_error_unavailable();
    }

    /* Print the smtp greeting */
    fprintf(stdout, "220 %s ESMTP\r\n", 
            LM_STRING_BUFFER(config_qmail.smtpgreeting));
    fflush(stdout);

    /* Read an smtp command from stdin */
    /* This function allocated the string to hold the command */
    if (smtp_init(config_qmail.timeoutsmtpd, msd_timeout_read, msd_write,   /* read, write */
                  msd_queue_init, msd_queue, msd_queue_cleanup, msd_helo_callback, msd_ehlo_callback,   /* helo, ehlo */
                  msd_mail_callback, msd_rcpt_callback, /* mail, rcpt  */
                  msd_data_callback, msd_auth_callback, /* data, auth */
                  NULL, msd_rset_callback,  /* vrfy, rset */
                  msd_help_callback, NULL,  /* help, noop */
                  msd_quit_callback,    /* quit */
                  msd_header_line_callback, msd_body_line_callback) != 0) { /* header, body line callbacks */
        SYSLOG((LOG_ERR, "error initializing smtp"));
        msd_error_unavailable();
    }

    /* Command loop */
    while (smtp_read_command(&smtp_cmd) >= 0) {
        if ((numcmd >= config.max_smtp_cmds) && (config.max_smtp_cmds > 0)) {
            fprintf(stdout,
                    "552 Maximum number of SMTP commands per connection exceeded (#4.5.3.1) \r\n");
            SYSLOG((LOG_WARNING, "maximum number of SMTP commands exceeded: %d",
                    numcmd));
            msd_quit_callback(1);
        }

        /* Parse and handle the command, checking for errors */
        /* This function frees the string holding the commmand */
        if (smtp_handle_command(smtp_cmd) != 0) {
            /* QMAIL */
            /* TODO Print proper message here instead of what qmail does? */
            SYSLOG((LOG_ERR, "error handling smtp command"));
            msd_error_unavailable();
        }
        LM_STRING_FREE(smtp_cmd);
        numcmd++;
    }

    /* Exit the program */
    SYSLOG((LOG_ERR, "error during smtp command processing...exiting"));
    msd_error_unavailable();    /* doesn't return */
    return 0;
}

/* SIGALRM handler, prints timeout message and exits the program */
void msd_sigalrm(int arg)
{
    timeout_flag = 1;
    fprintf(stdout, "451 timeout (#4.4.2\r\n");
    SYSLOG((LOG_NOTICE, "read timeout"));
    msd_quit_callback(1);
}

/* read a line from the given file descriptor with a timeout */
int msd_timeout_read(int fd, lm_string_t * line)
{
    lm_string_t newstr = NULL;
    lm_string_t tmplmstr = NULL;
    int offset, nbytes, done, errors, eof;
    char tmpstr[2];
    char c;

    if (LM_STRING_NEW(newstr, "") != 0) {
        msd_error_out_of_memory();
    }

    offset = 0;
    nbytes = -1;
    eof = errors = done = 0;
    while (!done) {

        /* Setup an alarm signal handler */
        timeout_flag = 0;
        signal(SIGALRM, msd_sigalrm);

        /* Enable the for the timeout */
        alarm(config_qmail.timeoutsmtpd);

        nbytes = read(fd, &c, 1);

        switch (nbytes) {
            case 0:            /* end of file */
                SYSLOG((LOG_NOTICE, "Premature end of input data"));
                eof = 1;
                done = 1;
                break;
            case 1:            /* successfully read 1 byte */

                /* if errors was set, we were trying to recover.
                 * if we got a byte, that means we did recover */
                errors = 0;

                /* LM_STRING_CAT needs an lm_string_t so we make c into one (tmplmstr) */
                tmpstr[0] = c;
                tmpstr[1] = '\0';
                if (LM_STRING_NEW(tmplmstr, tmpstr) != 0) {
                    msd_error_out_of_memory();
                }

                /* add c (which has been converted to a string) to the temporary
                 * line buffer (newstr) */
                if (LM_STRING_CAT(newstr, tmplmstr) != 0) {
                    msd_error_out_of_memory();
                }

                /* free memory used by the temporary lm_string_t */
                LM_STRING_FREE(tmplmstr);

                if (c == '\n') {
                    /* this marks the end of the line, so we are done reading a line */
                    done = 1;
                }

                break;
            case -1:           /* an error occurred with read() */
                SYSLOG((LOG_ERR, "temporary read() error: %s",
                        strerror(errno)));
                errors++;
                /* arbitrarily chose 3 (third time is the charm) */
                if (errors >= 3) {
                    /* too many errors, giving up */
                    SYSLOG((LOG_ERR, "fatal read() error: tried 3 times"));
                    done = 1;
                }
                break;
            default:           /* an unknown error occured */
                SYSLOG((LOG_ERR, "msd_timeout_read() unknown error [%d]",
                        nbytes));
                errors++;
                done = 1;
                break;
        }
    }

    if (eof != 0) {
        /* hit end of file */
        /* won't be able to write to the client, so exit quietly */
        msd_quit_callback(1);

    } else if (errors > 0) {
        /* stopped due to errors */
        /* print a message to the client warning them */
        fprintf(stdout, "451 read error\r\n");
        msd_quit_callback(1);

    } else {
        /* copy our temporary line variable's address to the line out-var */
        *line = newstr;
        if (*line == NULL) {
            /* failure */
            SYSLOG((LOG_ERR, "msd_timeout_read(): line failed"));
            /* don't know how to recover, so exit */
            fprintf(stdout, "451 unknown read error\r\n");
            msd_quit_callback(1);
        } else {
            /* success */
            return (0);
        }
    }

    /* should not get here */
    SYSLOG((LOG_ERR, "Reached unreachable code in msd_timeout_read()"));
    return (1);

}

/* Write the given string to the file descriptor */
/* TODO Add timeouts and other error checking ? */
int msd_write(int fd, lm_string_t line)
{
    int retval;

    /* TODO should probably do some error checking here */
    LM_STRING_ASSERT(line);
    if (LM_STRING_CHECK(line) != 0) {
        return -1;
    }

    LM_ASSERT(fd > 0);
    retval = write(fd, LM_STRING_BUFFER(line), LM_STRING_LEN(line));

    /* TODO Do we want an fsync here all the time? */
    fsync(fd);
    return retval == LM_STRING_LEN(line);
}
