/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: lmstring.c,v 1.33 2003/10/27 20:41:02 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

/* Standard C includes */
#define __USE_GNU
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

/* liblm includes */
#include <liblm.h>

/*
 * Acutal string buffer.  This may include the actual length of the string array
 * later so we don't have to do so many malloc's and free's but that's way off
 */
typedef struct {
    int length;
    char *buffer;
} tag_lm_string_t;

/* If _LMDEBUG is specified more information is passed to the malloc library
   so that information about what the string's symbol and allocation location 
   can be stored for later */
#if defined(_LMDEBUG)
#define LM_STRING_MALLOC(ptr, size) _lm_malloc(size, (void*)&ptr, sym, file, line)
#else                           /* defined(_LMDEBUG) */
#define LM_STRING_MALLOC(ptr, size) LM_MALLOC(ptr, size)
#endif                          /* defined(_LMDEBUG) */

/* Put this in just in case we want to make a custom allocator later which will be much faster */
#define __LM_STRING_FREE(ptr) LM_FREE(ptr)

size_t strnlen(const char *s, size_t maxlen);

/* Allocate a new string object */
#if defined(_LMDEBUG)
int _lm_string_new_debug(char *file, int line, char *sym, lm_string_t * pstr,
                         char *string)
#else
int _lm_string_new(lm_string_t * pstr, char *string)
#endif
{
    tag_lm_string_t *str;
    size_t size;
    int stringlen;

    LM_WARN(*pstr != NULL
            && printf("*** Warning for symbol %s, file %s, line %d...\n", sym,
                      file, line));
    LM_WARN(string == NULL);

    /* Calculate the length of the string from the init string */
    stringlen = strnlen(string, LM_STRING_MAX_WARN_SIZE);
    LM_WARN(stringlen >= LM_STRING_MAX_WARN_SIZE);
    if (stringlen >= LM_STRING_MAX_WARN_SIZE)
        return -1;

    /* Allocate the string structure */
    str = NULL;
    size = sizeof(*str) + stringlen + 1;
    if (LM_STRING_MALLOC(str, size)) {
        return -1;
    }
    LM_WARN(string == NULL);
    if (str == NULL)
        return -1;
    str->length = stringlen;

    /* Move the buffer pointer to after the end of the structure where 
       the string space is allocated */
    str->buffer = ((char *) str) + sizeof(*str);
    strncpy(str->buffer, string, str->length);
    str->buffer[str->length] = '\0';

    *pstr = (lm_string_t) str;
    LM_STRING_ASSERT(*pstr);

    return *pstr == NULL;
}

/* Free the memory allocated by the string */
int _lm_string_free(lm_string_t * pstr)
{
    tag_lm_string_t *str;

    /* Santiry checks */
    LM_ASSERT(pstr != NULL);
    str = (tag_lm_string_t *) * pstr;

    if (str == NULL)
        return 0;

    /* Check the integrity of the string */
    LM_STRING_ASSERT((lm_string_t) str);

    /* Free the memory */
    __LM_STRING_FREE(str);
    *pstr = (lm_string_t) str;

    return *pstr != NULL;
}

/* Return the length of the string */
int _lm_string_len(lm_string_t str)
{
    LM_STRING_ASSERT(str);
    return ((tag_lm_string_t *) str)->length;
}

char *_lm_string_buffer(lm_string_t str)
{
    LM_STRING_ASSERT(str);
    return ((tag_lm_string_t *) str)->buffer;
}

/* Check the integrity of the string */
#if defined(_LMDEBUG)
int _lm_string_check_debug(char *file, int line, lm_string_t str)
#else
int _lm_string_check(lm_string_t str)
#endif                          /* _LMDEBUG */
{
    int retval = 0;
    tag_lm_string_t *this;

    retval |= (str == NULL);

    this = (tag_lm_string_t *) str;

    /* Check the null terminator is where it should be */
    retval |= (this->buffer[this->length] != '\0');

    return retval;
}

#if defined(_LMDEBUG)
void _lm_string_assert(lm_string_t str, char *file, int line)
{
    tag_lm_string_t *this;
    int length;

    LM_WARN(str == NULL);
    this = (tag_lm_string_t *) str;

    /* Check the integrity of the allocated memory */
    LM_MEMASSERT(this);

    /* Check the size to make sure it's ok */
    LM_WARN(this->length > LM_STRING_MAX_WARN_SIZE);

    /* Make sure the length of the string is consistent with the stored size */
    length = strnlen(this->buffer, this->length);
    LM_WARN(length != this->length);

    /* Perform other run time checks and warn if fail */
    LM_WARN(_lm_string_check_debug(file, line, str) != 0);
}
#endif                          /* defined(_LMDEBUG) */

#if defined(_LMDEBUG)
int _lm_string_dup_debug(char *file, int line, char *sym, lm_string_t * ptostr,
                         lm_string_t fromstr)
#else
int _lm_string_dup(lm_string_t * ptostr, lm_string_t fromstr)
#endif
{
    size_t size;
    tag_lm_string_t *newstr;

    newstr = NULL;

    /* Warn if string to copy to isn't null since that means that it's already allocated */
    LM_WARN(*ptostr != NULL);
    LM_STRING_ASSERT(fromstr);

    size = sizeof(*newstr) + ((tag_lm_string_t *) fromstr)->length + 1;
    LM_WARN(size < 0);
    LM_WARN(size > LM_STRING_MAX_WARN_SIZE);

    if (LM_STRING_MALLOC(newstr, size))
        return -1;

    newstr->buffer = ((char *) newstr) + sizeof(*newstr);
    newstr->length = ((tag_lm_string_t *) fromstr)->length;
    memcpy(newstr->buffer, ((tag_lm_string_t *) fromstr)->buffer,
           ((tag_lm_string_t *) fromstr)->length + 1);

    /* Need to update the buffer pointer to point at our string and not the duplicated string's buffer */
    LM_STRING_ASSERT((lm_string_t) newstr);

    *ptostr = (lm_string_t) newstr;

    return *ptostr == NULL;
}

#if defined(_LMDEBUG)
int _lm_string_cat_debug(char *file, int line, char *sym, lm_string_t * ptostr,
                         lm_string_t fromstr)
#else
int _lm_string_cat(lm_string_t * ptostr, lm_string_t fromstr)
#endif
{
    size_t stringlen;
    tag_lm_string_t *tostr;
    tag_lm_string_t *newstr;

    stringlen = 0;

    LM_WARN(*ptostr == NULL);
    tostr = (tag_lm_string_t *) (*ptostr);

    /* Check that the strings are ok */
    LM_STRING_ASSERT((lm_string_t) tostr);
    LM_STRING_ASSERT(fromstr);

    /* Get the length of the new string */
    stringlen += tostr->length;
    stringlen += ((tag_lm_string_t *) fromstr)->length;
    LM_WARN(stringlen > LM_STRING_MAX_WARN_SIZE);

    /* Allocate a new string */
    newstr = NULL;
    if (LM_STRING_MALLOC(newstr, stringlen + 1 + sizeof(*newstr)))
        return -1;
    newstr->buffer = ((char *) newstr) + sizeof(*newstr);
    newstr->length = stringlen;

    /* Copy the old string into the new string */
    strncpy(newstr->buffer, tostr->buffer, tostr->length + 1);

    /* Copy the string to be cat'd to the end of the new string */
    strncpy(&newstr->buffer[tostr->length],
            ((tag_lm_string_t *) fromstr)->buffer,
            ((tag_lm_string_t *) fromstr)->length + 1);

    /* Free the old string */
    __LM_STRING_FREE(tostr);

    /* Assign the new string to the return string */
    *ptostr = (lm_string_t) newstr;

    return *ptostr == NULL;
}

int _lm_string_cmp(lm_string_t str1, lm_string_t str2)
{
    LM_STRING_ASSERT(str1);
    LM_STRING_ASSERT(str2);

    return strncmp(LM_STRING_BUFFER(str1), LM_STRING_BUFFER(str2),
                   LM_STRING_LEN(str1) + 1);
}

int _lm_string_casecmp(lm_string_t str1, lm_string_t str2)
{
    LM_STRING_ASSERT(str1);
    LM_STRING_ASSERT(str2);

    return strncasecmp(LM_STRING_BUFFER(str1), LM_STRING_BUFFER(str2),
                       LM_STRING_LEN(str1) + 1);
}

#if defined(_LMDEBUG)
int _lm_string_fgets_debug(char *file, int line, char *var, lm_string_t * pstr,
                           FILE * infile, int maxbytes, size_t * bytesread)
#else                           /* defined(_LMDEBUG) */
int _lm_string_fgets(lm_string_t * pstr, FILE * infile, int maxbytes,
                     size_t * bytesread)
#endif                          /* defined(_LMDEBUG) */
{
    char *fgets_buffer = NULL;
    size_t fgets_buffer_size = 0;
    lm_string_t newstr = NULL;
    char *read;

    LM_ASSERT(pstr != NULL);
    LM_WARN(*pstr != NULL);

    /* Allocate the fgets buffer on the stack */
    fgets_buffer_size = maxbytes;
    fgets_buffer = alloca(fgets_buffer_size);
    if (fgets_buffer == NULL)
        return -1;

    /* Read a line from the given input file */
    read = fgets(fgets_buffer, fgets_buffer_size, infile);
    LM_ASSERT(bytesread != NULL);
    *bytesread = read != NULL;

    /* If anything was read */
    if (*bytesread) {
        /* Create a new string object from the line that was read from the file */
#if defined(_LMDEBUG)
        _lm_string_new_debug(file, line, var, &newstr, fgets_buffer);
#else                           /* defined(_LMDEBUG) */
        LM_STRING_NEW(newstr, fgets_buffer);
#endif                          /* defined(_LMDEBUG) */
    } else {
        /* Just create an empty string since nothing was read from the file */
        LM_STRING_NEW(newstr, "");
    }

    /* Assign the new string object to the return string pointer */
    *pstr = newstr;

    /* Return succses if everything went ok */
    return *pstr == NULL;
}

int _lm_string_to_int(lm_string_t str, int *retval)
{
    int tmp;
    int ret;

    LM_STRING_ASSERT(str);
    LM_ASSERT(retval != NULL);

    ret = sscanf(LM_STRING_BUFFER(str), "%d", &tmp);
    /* If we read anything from the string that we asked for assign to the variable given */
    if (ret == 1) {
        *retval = tmp;
    }

    /* If nothing was read return error, else sucesses */
    return ret == 0 ? -1 : 0;
}

/* Remove whitespace from the end of a string and re null terminate */
int _lm_string_chop(lm_string_t str)
{
    size_t index;
    tag_lm_string_t *string;

    LM_STRING_ASSERT(str);
    string = (tag_lm_string_t *) str;

    /* We won't bother reallocating the string here since the string
       is shrinking.
       Might put in a check to if the string shrinks masive amounts and then 
       realloc, but that's not needed right now */

    /* Don't do anything if string length is zero */
    if (string->length > 0) {
        /* Now get rid of any spaces that might be on the end */
        index = string->length;

        while (--index >= 0) {
            if (string->buffer[index] != ' ' && string->buffer[index] != '\r'
                && string->buffer[index] != '\n'
                && string->buffer[index] != '\t') {
                break;
            }
        }

        index++;
        LM_ASSERT(index >= 0 && index <= string->length);
        string->buffer[index] = '\0';
        string->length = index;
    }

    LM_STRING_ASSERT(str);
    return 0;
}

int _lm_string_chop_head(lm_string_t str)
{
    size_t index;
    tag_lm_string_t *string;

    LM_STRING_ASSERT(str);
    string = (tag_lm_string_t *) str;

    /* We won't bother reallocating the string here since the string
       is shrinking.
       Might put in a check to if the string shrinks masive amounts and then 
       realloc, but that's not needed right now */

    /* Now get rid of any spaces that might be on the end */
    index = 0;
    /* Don't do anything if string length is zero */
    if (string->length > 0) {
        while (index < string->length) {
            if (string->buffer[index] != ' ' && string->buffer[index] != '\r'
                && string->buffer[index] != '\n'
                && string->buffer[index] != '\t') {
                break;
            }
            index++;
        }

        LM_ASSERT(index >= 0 && index <= string->length);
        string->buffer += index;
        string->length -= index;
    }
    LM_STRING_ASSERT(str);
    return 0;
}

int _lm_string_chr(lm_string_t str, char chr, int *index)
{
    tag_lm_string_t *string;
    int tmp;

    LM_STRING_ASSERT(str);
    string = (tag_lm_string_t *) str;
    LM_ASSERT(index);

    tmp = 0;

    /* Search the string for the character */
    while ((tmp < string->length) && (string->buffer[tmp] != chr)) {
        tmp++;
    }

    /* If the index is less than the string length we found it */
    if (tmp < string->length) {
        LM_ASSERT(string->buffer[tmp] == chr);
        *index = tmp;
        return 0;
    }

    /* Return error if the character isn't found */
    *index = -1;
    return -1;
}

int _lm_string_rchr(lm_string_t str, char chr, int *index)
{
    tag_lm_string_t *string;
    int tmp;

    LM_STRING_ASSERT(str);
    string = (tag_lm_string_t *) str;
    LM_ASSERT(index);

    tmp = string->length;

    /* Search the string for the character */
    while (string->buffer[tmp] != chr && tmp >= 0)
        tmp--;

    /* If the index is greater than zero the string length we found it */
    if (tmp >= 0) {
        LM_ASSERT(string->buffer[tmp] == chr);
        *index = tmp;
        return 0;
    }

    /* Return error if the character isn't found */
    *index = -1;
    return -1;
}

int _lm_string_start_shift(lm_string_t str, int amount)
{
    tag_lm_string_t *string;

    LM_STRING_ASSERT(str);
    string = (tag_lm_string_t *) str;

    LM_WARN(amount > string->length);

    string->buffer += amount;
    string->length -= amount;

    LM_STRING_ASSERT(str);

    return 0;
}

int _lm_string_end_shift(lm_string_t str, int amount)
{
    tag_lm_string_t *string;

    LM_STRING_ASSERT(str);
    string = (tag_lm_string_t *) str;

    LM_WARN(amount > string->length);

    string->length -= amount;
    string->buffer[string->length] = '\0';
    LM_STRING_ASSERT(str);

    return 0;
}

int _lm_string_get_line(lm_string_t str, lm_string_t * line)
{
    lm_string_t newline = NULL;
    char *buffer = NULL;
    int index;

    LM_STRING_ASSERT(str);

    buffer = LM_STRING_BUFFER(str);
    if (!LM_STRING_CHR(str, '\n', index)) {
        buffer[index] = '\0';
        if (LM_STRING_NEW(newline, buffer)) {
            /* Out of memory return error */
            return -1;
        }

        buffer[index] = '\n';
        index++;
    } else {
        if (LM_STRING_DUP(newline, str)) {
            /* Out of memory, return error */
            return -1;
        }
        index = LM_STRING_LEN(str);
        LM_STRING_CHOP(newline);
    }

    /* Shift off the start of the string to get rid of the line we just found */
    LM_STRING_START_SHIFT(str, index);

    LM_WARN(*line != NULL);
    LM_ASSERT(newline != NULL);
    *line = newline;
    return *line == NULL;
}

int _lm_string_lower(lm_string_t str)
{
    int i;
    char *buffer;

    LM_STRING_ASSERT(str);

    buffer = LM_STRING_BUFFER(str);

    for (i = LM_STRING_LEN(str); i > 0; i--) {
        *buffer = tolower(*buffer);
        buffer++;
    }

    return 0;
}

int _lm_string_upper(lm_string_t str)
{
    int i;
    char *buffer;

    LM_STRING_ASSERT(str);

    buffer = LM_STRING_BUFFER(str);

    for (i = LM_STRING_LEN(str); i > 0; i--) {
        *buffer = toupper(*buffer);
        buffer++;
    }

    return 0;
}

int _lm_string_replace(lm_string_t str, char from, char to)
{
    int i;
    char *buffer;

    LM_STRING_ASSERT(str);

    buffer = LM_STRING_BUFFER(str);

    for (i = LM_STRING_LEN(str); i > 0; i--) {
        if (*buffer == from)
            *buffer = to;
        buffer++;
    }

    return 0;
}
