/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: lmconfig_control.c,v 1.20 2003/10/23 00:03:52 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

/* C includes */
#include <stdlib.h>
#include <stdio.h>
#include <sys/stat.h>

/* liblm includes */
#include <liblm.h>

/* Control type variables and state  */
static lm_string_t config_control_dir;
static lm_string_t newline;
static lm_string_t true, false;

static int __lm_config_control_open(void);
static int __lm_config_control_close(void);
static int __lm_config_control_read_string(lm_config_key_t key,
                                           lm_config_string_t * pstring,
                                           lm_config_string_t defval);
static int __lm_config_control_read_number(lm_config_key_t key,
                                           lm_config_number_t * pnumber,
                                           lm_config_number_t defval);
static int __lm_config_control_read_bool(lm_config_key_t key,
                                         lm_config_bool_t * pbool,
                                         lm_config_bool_t defval);
static int __lm_config_control_write_string(lm_config_key_t key,
                                            lm_config_string_t string);
static int __lm_config_control_write_number(lm_config_key_t key,
                                            lm_config_number_t number);
static int __lm_config_control_write_bool(lm_config_key_t key,
                                          lm_config_bool_t bool);

static int __lm_config_control_read_controlfile(lm_config_key_t key,
                                                lm_config_string_t * pstring);

int _lm_config_control_set_param(lm_string_t controldir)
{
    /* Check paramters */
    LM_STRING_ASSERT(controldir);

    /* Check to be sure the parmeters arent' already set */
    LM_WARN(config_control_dir != NULL);

    config_control_dir = NULL;
    LM_STRING_DUP(config_control_dir, controldir);

    _lm_config_set_functions(__lm_config_control_open,
                             __lm_config_control_close,
                             __lm_config_control_read_string,
                             __lm_config_control_read_number,
                             __lm_config_control_read_bool,
                             __lm_config_control_write_string,
                             __lm_config_control_write_number,
                             __lm_config_control_write_bool);

    /* Attempt to open a control type config */
    return __lm_config_control_open();
}

static int __lm_config_control_open(void)
{
    struct stat buf;

    /* Sanity check paramters */
    LM_STRING_ASSERT(config_control_dir);

    /* Check to see if the control directory exists */
    if (stat(LM_STRING_BUFFER(config_control_dir), &buf)) {
        SYSLOG((LOG_ERR, "Control direcory %s does not exist!",
                LM_STRING_BUFFER(config_control_dir)));
        LM_STRING_FREE(config_control_dir);
        return -1;
    }

    LM_STRING_NEW(newline, "\n");
    LM_STRING_NEW(true, "true");
    LM_STRING_NEW(false, "false");

    return 0;
}

static int __lm_config_control_close(void)
{
    /* Free the configuration strings since they have to use *_set_param to reopen 
       the configuration database */

    LM_STRING_FREE(config_control_dir);
    LM_STRING_FREE(newline);
    LM_STRING_FREE(true);
    LM_STRING_FREE(false);
    return 0;
}

static int __lm_config_control_read_string(lm_config_key_t key,
                                           lm_config_string_t * pstring,
                                           lm_config_string_t defval)
{
    lm_string_t newstr;

    newstr = NULL;
    if (__lm_config_control_read_controlfile(key, &newstr)) {
        LM_ASSERT(newstr == NULL);
        SYSLOG((LOG_DEBUG,
                "Couldn't open control file for reading...using default string value"));
        LM_STRING_DUP(newstr, defval);
    }

    LM_STRING_CHOP(newstr);

    /* If a string was already allocated in pstring free it */
    if (*pstring != NULL) {
        lm_string_t tmpstring;

        tmpstring = *pstring;
        LM_STRING_FREE(tmpstring);
    }
    *pstring = newstr;

    return newstr == NULL;
}

static int __lm_config_control_read_number(lm_config_key_t key,
                                           lm_config_number_t * pnumber,
                                           lm_config_number_t defval)
{
    lm_string_t newstr;
    int readval;

    newstr = NULL;
    if (__lm_config_control_read_controlfile(key, &newstr)) {
        LM_ASSERT(newstr == NULL);
        SYSLOG((LOG_DEBUG,
                "Couldn't open control file for reading...using default number value"));
        *pnumber = defval;
    } else {
        /* Control file was read sucessfully, now convert the number from text to binary */
        if (LM_STRING_TO_INT(newstr, readval)) {
            SYSLOG((LOG_DEBUG,
                    "Asking for number from control file which does not contain a number...using default!"));
            *pnumber = defval;
        } else {
            *pnumber = readval;
        }
        LM_STRING_FREE(newstr);
    }

    return 0;
}

static int __lm_config_control_read_bool(lm_config_key_t key,
                                         lm_config_bool_t * pbool,
                                         lm_config_bool_t defval)
{
    lm_string_t newstr;

    newstr = NULL;
    if (__lm_config_control_read_controlfile(key, &newstr)) {
        LM_ASSERT(newstr == NULL);
        SYSLOG((LOG_DEBUG,
                "Couldn't open control file for reading...using default boolean value"));
        *pbool = defval;
    } else {
        /* Remove newline */
        LM_STRING_CHOP(newstr);

        /* Control file was read sucessfully, now convert the boolean from text to binary */
        if (!LM_STRING_CASECMP(newstr, true))
            *pbool = 1;
        else if (!LM_STRING_CASECMP(newstr, false))
            *pbool = 0;
        else {
            SYSLOG((LOG_DEBUG,
                    "Asking for boolean from control file which does not contain either true or false [%s]...using default!", LM_STRING_BUFFER(newstr)));
            *pbool = defval;
        }

        LM_STRING_FREE(newstr);
    }

    return 0;
}

static int __lm_config_control_write_string(lm_config_key_t key,
                                            lm_config_string_t string)
{
    SYSLOG((LOG_WARNING, "CONTROL WRITE STRING NOT IMPLEMENTED!!!"));
    return -1;
}

static int __lm_config_control_write_number(lm_config_key_t key,
                                            lm_config_number_t number)
{
    SYSLOG((LOG_WARNING, "CONTROL WRITE NUMBER NOT IMPLEMENTED!!!"));
    return -1;
}

static int __lm_config_control_write_bool(lm_config_key_t key,
                                          lm_config_bool_t bool)
{
    SYSLOG((LOG_WARNING, "CONTROL WRITE BOOL NOT IMPLEMENTED!!!"));
    return -1;
}

/* TODO Handle multi line files properly by adding newline or space seperators after each line is read */
/* TODO this only handles commented lines with the comment character on in the first column.
   Need to check if control files allow for comments anywhere on the line */
static int __lm_config_control_read_controlfile(lm_config_key_t key,
                                                lm_config_string_t * pstring)
{
    FILE *file;
    int bytesread;
    lm_string_t newstr = NULL;
    lm_string_t filename = NULL;
    lm_string_t contents = NULL;

    LM_STRING_ASSERT(key);
    LM_WARN(*pstring != NULL);

    /* If the control directory hasn't been set or didn't exist when the config 
       was open return error so that the default value will be set */
    if (config_control_dir == NULL) {
        SYSLOG((LOG_ERR,
                "control directory not set...returning error to set default"));
        return -1;
    }

    LM_STRING_DUP(filename, config_control_dir);
    LM_STRING_CAT(filename, key);

    file = fopen(LM_STRING_BUFFER(filename), "r");
    if (file != NULL) {
        LM_STRING_NEW(contents, "");

        do {
            if (LM_STRING_FGETS(newstr, file, 1024, bytesread)) {
                SYSLOG((LOG_ERR, "error reading string from control file"));
            } else {
                if (bytesread) {
                    int cindex;
                    char *tmp;

                    tmp = LM_STRING_BUFFER(newstr);

                    /* Remove comments from the line */
                    if (!LM_STRING_CHR(newstr, '#', cindex)) {
                        /* We found a comment character, shift it off the
                         * end of the string 
                         */
                        LM_STRING_END_SHIFT(newstr,
                                            LM_STRING_LEN(newstr) - cindex);
                    }

                    /* Strip the whitespace from the end of the line */
                    LM_STRING_CHOP(newstr);

                    /* If we have a non blank line */
                    if (LM_STRING_LEN(newstr) > 0) {
                        /* Put a single newline at the end of the line */
                        LM_STRING_CAT(newstr, newline);

                        /* Save the contents of the line to the contets string  */
                        LM_STRING_CAT(contents, newstr);
                    }
                }
            }

            LM_STRING_FREE(newstr);
        } while (bytesread != 0);

        LM_STRING_ASSERT(contents);
        LM_ASSERT(*pstring == NULL);

        *pstring = contents;

        fclose(file);
    }

    LM_STRING_FREE(filename);

    return *pstring == NULL;
}
