/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: lmalloc.c,v 1.13 2003/10/27 20:41:02 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

/* Standard C includes */
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* LM library includes */
#include <liblm.h>

/* Only include this in library if requested */
#if defined (_LMDEBUG) && defined(LM_USE_ALLOC)

struct tag_lmalloc_t {
    void *ptr;                  /* pointer to beginning of memory block */
    size_t size;                /* size of the block in bytes */
    int line;                   /* line block was allocated on */
    char *file;                 /* file block was allocated from */
    char *sym;                  /* variable's symbol */
    unsigned long *tail;
    struct tag_lmalloc_t *next;
    struct tag_lmalloc_t *prev;
};

typedef struct tag_lmalloc_t lmalloc_t;

static lmalloc_t *mem_head;
static lmalloc_t *mem_prev;
static int mem_num_blocks;

typedef void *(*alloc_func) (size_t);

/* Internal function to take one argument so we can pass it to _lm_alloc */
static void *__lm_calloc(size_t size)
{
    return calloc(1, size);
}

static int _lm_alloc(size_t size, void **pptr, char *sym, char *file, int line,
                     alloc_func func)
{
    size_t fullsize;
    lmalloc_t head;
    void *mem;

    /* Clear the head structure */
    memset(&head, 0, sizeof(head));

    /* Check for sane size asked for */
    LM_WARN((size <= 0) && fprintf(stderr, "** symbol: %s\n", sym));

    /* Warn if allocation requested is greater than LM_ALLOC_MAX_WARN_SIZE */
    LM_WARN((size > LM_ALLOC_MAX_WARN_SIZE)
            && fprintf(stderr, "** symbol: %s\n", sym));

    /* Pointers passed in should be initialized to null just to inforce good coding 
       pracitices and to prevent realloction over already allocted pointers */
    LM_ASSERT(pptr != NULL);
    LM_WARN((*pptr != NULL)
            && fprintf(stderr, "** symbol: %s (*pptr => 0x%x)\n", sym,
                       (unsigned int) (*pptr)));

    /* Calclulate the full size of the block with head and tail added on */
    fullsize = size + 2 * sizeof(head);

    /* Allocate the block normally */
    mem = func(fullsize);

    if (mem == NULL) {
        /* Warn on failed allocation to make sure they get it but 
           don't do any error handling here since it will be compiled out
           in release and we want to have error checking on these things all the time */
        LM_WARN((mem == NULL) && fprintf(stderr, "** symbol: %s\n", sym));
        return -1;
    }

    /* Fill in the header fields */
    head.ptr = (char *) mem + sizeof(head);
    head.size = size;
    head.line = line;
    head.file = file;
    head.sym = sym;

    /* First allocation by program */
    if (mem_head == NULL) {
        LM_WARN(mem_prev != NULL);

        mem_head = mem;
        mem_head->prev = NULL;
        mem_head->next = NULL;
    } else {
        LM_WARN(mem_prev == NULL);

        head.prev = mem_prev;
        mem_prev->next = mem;
        head.next = NULL;
    }

    /* Save the pointer the current block for the next allocation */
    mem_prev = mem;

    /* Put a signature on the end of the block to check for overruns */
    head.tail = (unsigned long *) ((char *) head.ptr + head.size);
    *head.tail = (unsigned long) head.ptr;

    /* Fill in the structures on the newly allocated block */
    *(lmalloc_t *) mem = head;

    /* Increment the number of allocated blocks */
    mem_num_blocks++;

    /* Return the pointer to the calling function */
    *pptr = head.ptr;
    return head.ptr == NULL;
}

int _lm_malloc(size_t size, void **pptr, char *sym, char *file, int line)
{
    return _lm_alloc(size, pptr, sym, file, line, malloc);
}

int _lm_calloc(size_t size, void **pptr, char *sym, char *file, int line)
{
    return _lm_alloc(size, pptr, sym, file, line, __lm_calloc);
}

/* Free a block of memory and remove it from the linked list of blocks */
void _lm_free(void **pptr, char *expr, char *file, int line)
{
    lmalloc_t *head, *prev, *next;
    void *ptr;

    LM_WARN(pptr == NULL);
    ptr = *pptr;

    LM_WARN((ptr == NULL)
            && fprintf(stderr, "*** symbol %s, file %s, line %d\n", expr, file,
                       line));
    LM_MEMASSERT(ptr);

    /* Get the structure for the head and tail pointer */
    head = (lmalloc_t *) ((char *) ptr - sizeof(*head));

    /* Remove this block from the linked list of blocks */
    prev = head->prev;
    next = head->next;

    /* Move the pointers to the next and previous blocks to remove this one */
    if (next)
        next->prev = prev;
    if (prev)
        prev->next = next;

    /* If we were the last block, null out the list */
    mem_num_blocks--;
    if (mem_num_blocks == 0) {
        mem_head = NULL;
        mem_prev = NULL;
    }
    if (mem_head == head) {
        mem_head = head->next;
    }
    if (mem_prev == head) {
        mem_prev = head->prev;
    }

    /* Put some crap in the header, allocated data  structure and tail sig */
    memset(head, 0xDEADBEEF, sizeof(head) + head->size);

    /* Free the allocated memory block */
    free(head);

    /* Null out the pointer so the next reference will be caught */
    *pptr = NULL;

    return;
}

void _lm_memassert(void *ptr, char *expr, char *file, int line)
{
    lmalloc_t *head;

    if (ptr == NULL) {
        fprintf(stderr,
                "LM_MEMASSERT FAIL! NULL pointer (could be already freed) \"%s\", file %s, line %d\n",
                expr, file, line);
        fflush(stderr);
        LM_WARN(ptr == NULL);
    }

    head = (lmalloc_t *) ((char *) ptr - sizeof(*head));

    if (head->tail == NULL) {
        fprintf(stderr,
                "LM_MEMASSERT FAIL! NULL Tail pointer \"%s\", file %s, line %d\n",
                expr, file, line);
        fflush(stderr);
        LM_WARN(head->tail == NULL);
    }

    /* Check to see if the tail signature has been trashed by an overrun */
    if ((unsigned long) head->ptr != *(head->tail)) {
        fprintf(stderr,
                "LM_MEMASSERT FAIL! Tail signature not correct for \"%s\"!, file %s, line %d\n",
                expr, file, line);
        fflush(stderr);
        LM_WARN((unsigned long) head->ptr != *(head->tail));
    }

    fflush(stderr);
    return;
}

void _lm_meminfo(FILE * outfile, void *ptr, char *expr, char *file, int line)
{
    lmalloc_t *info;

    info = (lmalloc_t *) ((char *) ptr - sizeof(lmalloc_t));

    fprintf(outfile, "\nMemory Info for variable %s, file %s, line %d\n", expr,
            file, line);
    fprintf(outfile, "Allocated symbol name: %s\n", info->sym);
    fprintf(outfile, "Allocation info: file %s, line %d\n", info->file,
            info->line);
    fprintf(outfile, "Address: 0x%x (0x%x acutal)\n", (unsigned int) info->ptr,
            (unsigned int) info);
    fprintf(outfile, "Size: %d\n", info->size);
    fprintf(outfile, "Next/Prev pointers: 0x%x/0x%x\n",
            (unsigned int) info->next, (unsigned int) info->prev);
    fprintf(outfile, "Tail pointer: 0x%x\n", (unsigned int) info->tail);
    fflush(outfile);
    return;
}

void _lm_memstat(FILE * outfile, char *file, int line)
{
    lmalloc_t *cur;
    int numblocks;
    int totalbytes;

    cur = mem_head;
    numblocks = 0;
    totalbytes = 0;

    fprintf(outfile, "\nLM MEMORY STATS, file %s, line %d\n", file, line);
    fflush(outfile);

    /* Go through all the blocks in the list */
    while (cur != NULL) {
        /* Do a check on all the blocks in the list */
        _lm_memassert(cur->ptr, cur->sym, cur->file, cur->line);

        /* Count the number of allocated blocks */
        numblocks++;

        /* Sum the total number of bytes allocated */
        totalbytes += cur->size;

        /* Move to the next block in the list */
        cur = cur->next;
    }

    LM_WARN(numblocks != mem_num_blocks);
    fprintf(outfile, "Blocks allocated: %d\n", numblocks);
    fprintf(outfile, "Total memory allocated: %d bytes\n", totalbytes);
    fprintf(outfile, "LM_ALLOC overhead: %d bytes\n\n",
            numblocks * sizeof(lmalloc_t) * 2);

    fflush(outfile);
    return;
}

void _lm_memdump(FILE * outfile, char *file, int line)
{
    lmalloc_t *cur;
    int numblocks;

    cur = mem_head;
    numblocks = 0;

    fprintf(outfile, "\nLM MEMORY DUMP, file %s, line %d\n", file, line);
    fflush(outfile);

    /* Go through all the blocks in the list */
    while (cur != NULL) {
        /* Do a check on all the blocks in the list */
        _lm_memassert(cur->ptr, cur->sym, cur->file, cur->line);

        _lm_meminfo(outfile, cur->ptr, cur->sym, cur->file, cur->line);

        numblocks++;

        /* Move to the next block in the list */
        cur = cur->next;
    }

    LM_WARN(numblocks != mem_num_blocks);
    if (numblocks == 0)
        fprintf(outfile, "No blocks allocated!\n");

    fflush(outfile);
    return;
}

#endif                          /* defined(_LMDEBUG) && defined(_LM_USE_ALLOC) */
