/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Josh Wilsdon <josh@wizard.ca>
 *
 * CVS Id: $Id: libgetbdb.c,v 1.7 2003/10/22 22:34:08 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

#include <ctype.h>
#include <sys/wait.h>
#include <assert.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <db.h>
#include "libgetbdb.h"

static int findUser(DB * dbp, char *user);
static void strlower(char *buffer);

int getData(DB * dbp, unsigned int dbkeylen, char *dbkey,
            unsigned int *dbdatalen, char **dbdata)
{
    DBT key, data;
    int ret;

    if ((dbp == NULL) || (dbkey == NULL) || (dbkeylen == 0)) {
        return (-1);
    }

    /* Initialize key/data */
    memset(&key, 0, sizeof(key));
    memset(&data, 0, sizeof(data));

    key.data = dbkey;
    key.size = dbkeylen;

    /* Retrieve a key/data pair. */
    if ((ret = dbp->get(dbp, NULL, &key, &data, 0)) == 0) {
        /* found */
        if (dbdatalen != NULL) {
            *dbdatalen = data.size;
        }
        if (dbdata != NULL) {
            *dbdata = data.data;
        }
        return (0);
    }

    if (ret == DB_NOTFOUND) {
        /* not found */
        return (1);
    }

    syslog(LOG_ERR, "getData(): unhandled error [%d]\n", ret);

    return (-1);
}

int getKeyMangledData(DB * dbp, char *dbkey, char manglechar,
                      unsigned int *dbdatalen, char **dbdata)
{
    unsigned int len;
    int ret;

    if ((dbkey == NULL) || ((len = (unsigned int) strlen(dbkey)) == 0)) {
        return (-1);
    }

    dbkey[len] = manglechar;
    ret = getData(dbp, len + 1, dbkey, dbdatalen, dbdata);
    dbkey[len] = '\0';

    return (ret);
}

int getUserData(DB * dbp, char *user, unsigned int *dbdatalen, char **dbdata)
{
    /* need to append 'U' to string, so use getKeyMangledData() */
    return (getKeyMangledData(dbp, user, 'U', dbdatalen, dbdata));
}

int getUserStruct(DB * dbp, char *user, user_record_t * user_rec)
{
    unsigned int len;
    char *uidstr, *gidstr, *data, *ptr, *password, *dir, *quota;
    int ret, uid, gid;

    ret = getUserData(dbp, user, &len, &data);
    if (ret != 0) {
        return (ret);
    }

    if (len == 0) {
        /* bad data */
        return (-1);
    }

    ptr = data;
    password = ptr;
    ptr += strlen(password);
    ptr++;

    if ((ptr - data) >= len) {
        /* ran out of data */
        return (-1);
    }

    uidstr = ptr;
    ptr += strlen(uidstr);
    ptr++;

    if ((ptr - data) >= len) {
        /* ran out of data */
        return (-1);
    }

    gidstr = ptr;
    ptr += strlen(gidstr);
    ptr++;

    if ((ptr - data) >= len) {
        /* ran out of data */
        return (-1);
    }

    dir = ptr;
    ptr += strlen(dir);
    ptr++;

    if ((ptr - data) >= len) {
        /* ran out of data */
        return (-1);
    }

    quota = ptr;

    // printf("password[%s] uid[%s] gid[%s] dir[%s] quota[%s]\n", password, uidstr, gidstr, dir, quota);

    uid = atoi(uidstr);
    if (uid < 0) {
        uid = 0;
    }
    gid = atoi(gidstr);
    if (gid < 0) {
        gid = 0;
    }
    user_rec->password = password;
    user_rec->uid = (unsigned) uid;
    user_rec->gid = (unsigned) gid;
    user_rec->dir = dir;
    user_rec->quota = quota;

    return (0);
}

int getAliasData(DB * dbp, char *alias, unsigned int *dbdatalen, char **dbdata)
{
    /* need to append 'A' to string, so use getKeyMangledData() */
    return (getKeyMangledData(dbp, alias, 'A', dbdatalen, dbdata));
}

int getSpamEmailID(DB * dbp, char *email, unsigned int *dbdatalen,
                   char **dbdata)
{
    /* need to append 'S' to string, so use getKeyMangledData() */
    return (getKeyMangledData(dbp, email, 'S', dbdatalen, dbdata));
}

int getVirusEmailID(DB * dbp, char *email, unsigned int *dbdatalen,
                    char **dbdata)
{
    /* need to append 'V' to string, so use getKeyMangledData() */
    return (getKeyMangledData(dbp, email, 'V', dbdatalen, dbdata));
}

int getSpamRules(DB * dbp, char *email_id, unsigned int *dbdatalen,
                 char **dbdata)
{
    unsigned int len;

    if ((email_id == NULL) || ((len = (unsigned int) strlen(email_id)) == 0)) {
        return (-1);
    }

    return (getData(dbp, len, email_id, dbdatalen, dbdata));
}

DB *openDBFile(char *filename)
{
    DB *dbp;
    int ret;

    if (filename == NULL) {
        return (NULL);
    }

    if ((ret = db_create(&dbp, NULL, 0)) != 0) {
        syslog(LOG_ERR, "db_create: %s\n", db_strerror(ret));
        return (NULL);
    }

    if ((ret = dbp->open(dbp, filename, NULL, DB_HASH, DB_RDONLY, 0664)) != 0) {
        syslog(LOG_ERR, "openDBFile(%s): %s", filename, db_strerror(ret));
        return (NULL);
    }

    return (dbp);
}

int closeDBFile(DB * dbp)
{
    int ret;

    if ((ret = dbp->close(dbp, 0)) != 0) {
        syslog(LOG_ERR, "closeDBFile(): %s\n", db_strerror(ret));
    }

    return (0);
}

int dumpData(unsigned int len, char *data)
{
    char *ptr;

    ptr = data;
    while ((unsigned int) (ptr - data) < len) {
        printf("%s\n", ptr);
        fflush(stdout);
        ptr += strlen(ptr);
        ptr++;
    }

    return (0);
}

void strlower(char *buffer)
{
    char *ptr;

    ptr = buffer;
    while (*ptr != '\0') {
        *ptr = tolower(*ptr);
        ptr++;
    }

    return;
}

/*
 *
 * RETURN
 *  
 * -1   error
 *  0   user exists (user will be in realuser)
 *  1   user does not exist
 *
 * SIDE EFFECTS
 *
 *  Memory will be allocated.  
 *
 *  If the return value is 0, realuser will be a pointer to allocated
 *  memory which must be free()'d by the caller.
 *
 */
int getRealUser(DB * dbp, char *user, char **realuser)
{
    char *buffer, *ptr, *appendage;
    int ret, len;

    if ((user == NULL) || (strlen(user) == 0)) {
        /* invalid parameters */
        return (-1);
    }

    buffer = malloc(sizeof(char) * (strlen(user) + strlen("-default") + 1));
    if (buffer == NULL) {
        /* out of memory */
        return (-1);
    }
    /* 
     * XXX: WARNING ANY ERROR RETURN PAST THIS POINT MUST BE PRECEDED BY free(buffer)
     */

    strcpy(buffer, user);

    /* ensure lowercase */
    strlower(buffer);

    /* first see if user exists as is */
    ret = findUser(dbp, buffer);
    if (ret == 0) {
        /* user exists */
        *realuser = buffer;
        return (0);
    } else if (ret == -1) {
        /* error */
        free(buffer);
        return (-1);
    }

    /* does not exist as user@domain so we need to search for -defaults */

    /* find last '@' */
    ptr = strrchr(buffer, '@');
    if (ptr == NULL) {
        /* does not have '@', so does not exist */
        free(buffer);
        return (1);
    }

    /* need to make copy since we'll wipe out buffer */
    appendage = strdup(ptr);
    if (appendage == NULL) {
        /* out of memory */
        free(buffer);
        return (-1);
    }

    *ptr = '\0';
    /* buffer is now just user-ext-ext... */

    // printf("BUFFER[%s] APPENDAGE[%s]\n", buffer, appendage);

    len = strlen(buffer);
    if ((len > 0) && (buffer[len - 1] == '-')) {
        /* ends with a '-', so not valid */
        free(buffer);
        return (1);
    }

    while ((ptr = strrchr(buffer, '-')) != NULL) {
        *ptr = '\0';
        strcat(buffer, "-default");
        strcat(buffer, appendage);
        // printf("BUFF[%s]\n", buffer);
        ret = findUser(dbp, buffer);
        if (ret == 0) {
            /* user exists */
            *realuser = buffer;
            return (0);
        } else if (ret == -1) {
            /* error */
            free(buffer);
            return (-1);
        }
        *ptr = '\0';
    }

    strcpy(buffer, "+default");
    strcat(buffer, appendage);

    ret = findUser(dbp, buffer);
    // printf("BUFF[%s]\n", buffer);
    if (ret == 0) {
        *realuser = buffer;
        return (0);
    } else if (ret == -1) {
        /* error */
        free(buffer);
        return (-1);
    }

    free(buffer);

    /* not found to exist, and no error reported, so does not exist */
    return (1);
}

int findUser(DB * dbp, char *user)
{
    char *data;
    unsigned int datalen;
    int retu, reta;

    if ((retu = getUserData(dbp, user, &datalen, &data)) == 0) {
        /* definitely exists */
        return (0);
    }

    if ((reta = getAliasData(dbp, user, &datalen, &data)) == 0) {
        /* definitely exists */
        return (0);
    }

    if ((reta == -1) || (reta == -1)) {
        /* there was an error */
        return (-1);
    }

    /* user does not exist */
    return (1);
}

#if 0
int main()
{
    int ret;
    char line[80], *realuser;
    DB *dbp;

    dbp = openDBFile(config.user_info_dbfile);
    if (dbp == NULL) {
        printf("openDBFile() failed\n");
        return (-1);
    }

    while (fgets(line, 80, stdin)) {
        if (line[strlen(line) - 1] == '\n') {
            line[strlen(line) - 1] = '\0';
        }
        ret = getRealUser(dbp, line, &realuser);
        if (ret == 0) {
            printf("user exists as [%s]\n", realuser);
        } else if (ret == 1) {
            printf("user does not exist %s\n", line);
        } else {
            printf("error determining if user exists\n");
        }
    }

    closeDBFile(dbp);

    exit(0);
}
#endif
