/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: config_spam_rule.c,v 1.48 2003/10/29 19:40:39 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

/*
  Routines to load global and user spam rule configs
*/
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>

#ifdef USE_LM_DATABASE
#include "postgresdb.h"
#include "queries.h"
#endif

#include <liblm.h>

#include "config_msd.h" 
#include "config_common.h"
#include "config_spam_rule.h"
#include "config_misc_rule.h"
#include "strlist.h"
#include "mod_dbfile.h"

#include "common/controls.h"
#include "common/load_controls.h"

extern config_t config;
extern config_type_t config_type_info[];

config_spam_rule_t global_spam_rules;

int config_spam_rule_init_defaults(config_spam_rule_t * rules);
int config_spam_rule_init_defaults_from_globals(config_spam_rule_t * rules, config_spam_rule_t * global_rules);
int config_spam_rule_load_lists(config_spam_rule_t * rules, config_spam_rule_t * global_rules);
int config_spam_rule_load_variables(config_spam_rule_t * rules);
int config_spam_rule_load_db(lm_string_t addr, config_spam_rule_t * rules);
static int config_spam_rule_load_globals(void);


int config_spam_rule_verify(config_spam_rule_t * rules)
{
    return 0;
}

int config_spam_rule_load_spamdir(config_spam_rule_t * rules, lm_string_t spamdir)
{
    int retval;
    struct stat statinfo;
    static lm_string_t slash = NULL;

    LM_STRING_NEW(slash, "/");
    if (slash == NULL) {
        return (-1);
    }

    if ((spamdir != NULL) && (stat(LM_STRING_BUFFER(spamdir), &statinfo) == 0) 
                          && (S_ISDIR(statinfo.st_mode) != 0)) {

        if (LM_STRING_CAT(spamdir, slash) != 0) {
            return -1;
        }
        LM_STRING_FREE(slash);

        /* load user's spam rules as control files */
        syslog(LOG_INFO, "Loading spam rule control files from %s", LM_STRING_BUFFER(spamdir));
        retval = LM_CONFIG_CONTROL_SET_PARAM(spamdir);
    } else {
        /* could not stat */
        syslog(LOG_INFO, "User spam directory %s does not exist or is not directory", LM_STRING_BUFFER(spamdir));
        return (-1);
    }

    /* Load the spam rules for the set config */
    if (retval == 0) {
        /* load the user rules and append global rules as per user configuration */
        retval = config_spam_rule_read(&global_spam_rules, rules);
    }
    /* Close the configuration that was opened */
    LM_CONFIG_CLOSE;

    return retval;
}

void config_spam_rule_print(config_spam_rule_t spam_rules)
{
    lm_string_t string;

    printf("all_global_rules: %d\n", spam_rules.all_global_rules);
    printf("spam_check: %d\n", spam_rules.spam_check);
    printf("smtp_check: %d\n", spam_rules.smtp_check);
    printf("virus_check: %d\n", spam_rules.virus_check);
    printf("delivery_check: %d\n", spam_rules.delivery_check);
    printf("smtp_blocking: %d\n", spam_rules.smtp_blocking);
    printf("require_full_addr: %d\n", spam_rules.require_full_addr);
    printf("block_mail_from_self: %d\n", spam_rules.block_mail_from_self);
    printf("block_ip_in_addr: %d\n", spam_rules.block_ip_in_addr);
    printf("block_non_printable: %d\n", spam_rules.block_non_printable);
    printf("require_me_in_dest: %d\n", spam_rules.require_me_in_dest);
    printf("valid_from_domain: %d\n", spam_rules.valid_from_domain);
    printf("valid_bounce: %d\n", spam_rules.valid_bounce);
    printf("require_helo: %d\n", spam_rules.require_helo);
    printf("valid_helo_domain: %d\n", spam_rules.valid_helo_domain);
    printf("mail_from_strict_addr_parse: %d\n", spam_rules.mail_from_strict_addr_parse);
    printf("check_ip_reverse_dns: %d\n", spam_rules.check_ip_reverse_dns);
    printf("use_global_from_whitelist: %d\n", spam_rules.use_global_from_whitelist);
    printf("use_global_from_blacklist: %d\n", spam_rules.use_global_from_blacklist);
    printf("use_global_helo_whitelist: %d\n", spam_rules.use_global_helo_whitelist);
    printf("use_global_helo_blacklist: %d\n", spam_rules.use_global_helo_blacklist);
    printf("use_global_country_whitelist: %d\n", spam_rules.use_global_country_whitelist);
    printf("use_global_country_blacklist: %d\n", spam_rules.use_global_country_blacklist);
    printf("use_global_ip_whitelist: %d\n", spam_rules.use_global_ip_whitelist);
    printf("use_global_ip_blacklist: %d\n", spam_rules.use_global_ip_blacklist);

    if (spam_rules.from_whitelist != NULL) {
        STRLIST_TO_STRING(string, spam_rules.from_whitelist);
        printf("from_whitelist [%s]\n", LM_STRING_BUFFER(string));
        LM_STRING_FREE(string);
    }

    if (spam_rules.helo_whitelist != NULL) {
        STRLIST_TO_STRING(string, spam_rules.helo_whitelist);
        printf("helo_whitelist [%s]\n", LM_STRING_BUFFER(string));
        LM_STRING_FREE(string);
    }

    if (spam_rules.ip_whitelist != NULL) {
        STRLIST_TO_STRING(string, spam_rules.ip_whitelist);
        printf("ip_whitelist [%s]\n", LM_STRING_BUFFER(string));
        LM_STRING_FREE(string);
    }

    if (spam_rules.from_blacklist != NULL) {
        STRLIST_TO_STRING(string, spam_rules.from_blacklist);
        printf("from_blacklist [%s]\n", LM_STRING_BUFFER(string));
        LM_STRING_FREE(string);
    }

    if (spam_rules.helo_blacklist != NULL) {
        STRLIST_TO_STRING(string, spam_rules.helo_blacklist);
        printf("helo_blacklist [%s]\n", LM_STRING_BUFFER(string));
        LM_STRING_FREE(string);
    }

    if (spam_rules.country_blacklist != NULL) {
        STRLIST_TO_STRING(string, spam_rules.country_blacklist);
        printf("country_blacklist [%s]\n", LM_STRING_BUFFER(string));
        LM_STRING_FREE(string);
    }

    if (spam_rules.ip_blacklist != NULL) {
        STRLIST_TO_STRING(string, spam_rules.ip_blacklist);
        printf("ip_blacklist [%s]\n", LM_STRING_BUFFER(string));
        LM_STRING_FREE(string);
    }

    if (spam_rules.required_header_list != NULL) {
        STRLIST_TO_STRING(string, spam_rules.required_header_list);
        printf("required_header_list [%s]\n", LM_STRING_BUFFER(string));
        LM_STRING_FREE(string);
    }

    return;
}

int config_spam_rule_load(lm_string_t addr, config_spam_rule_t * rules, lm_string_t spamdir)
{
    int ret = 0;
    int email_id;

    email_id = -1;

    LM_ASSERT(rules != NULL);
    LM_ASSERT(addr != NULL);

    /* load globals if we haven't already */
    config_spam_rule_load_globals();

#ifdef USE_LM_DATABASE
    if (config.use_database != 0) {
        ret = config_spam_rule_load_db(addr, rules);
        if (ret == 0) {
            syslog(LOG_DEBUG, "loaded spam rules from database");
            return (ret);
        } else {
            syslog(LOG_DEBUG, "loading spam rules from database failed");
        }

        /* pass through to next method */
    }
#endif

#ifdef USE_LM_DBFILE
    /* In all cases when we are not using Magicmail Database, we
     * want to check whether the user has spam checking enabled by
     * the administrator through the user.db dbfile.
     */
    email_id = getEmailIdDBFile(LM_STRING_BUFFER(addr));
    if (email_id != -1) {
        syslog(LOG_DEBUG, "User [%s] has spam checking enabled (Id: %d)", LM_STRING_BUFFER(addr), email_id);
    } else {
        syslog(LOG_DEBUG, "User [%s] does not have spam checking enabled by admin", LM_STRING_BUFFER(addr));
        return (-1);
    }
#endif 

    if (spamdir != NULL) {
        ret = config_spam_rule_load_spamdir(rules, spamdir);
        if (ret == 0) {
            syslog(LOG_DEBUG, "loaded spam rules from spamdir");
            return (ret);
        } else {
            syslog(LOG_DEBUG, "loading spam rules from spamdir failed");
        }

        /* pass through to next method */
    }

#ifdef USE_LM_DBFILE

    if (config.use_dbfile != 0) {
        ret = load_user_spam_rules(LM_STRING_BUFFER(addr), rules);
        if (ret == 0) {
            syslog(LOG_DEBUG, "loaded spam rules from dbfile");
            return (ret);
        } else {
            syslog(LOG_DEBUG, "loading spam rules from dbfile failed");
        }

        /* pass through to next method */
    }
#endif

    return (-1);    /* ERROR */
}

/* Load the configuration files for spam checking, appending the global rule lists 
 * if desired */
int config_spam_rule_read(config_spam_rule_t * global_rules,
                          config_spam_rule_t * rules)
{
    int retval = 0;

    LM_ASSERT(rules != NULL);
    /* Don't really need this since we know we will be filling in
       all of the variables in the structure */
    memset(rules, 0, sizeof(*rules));

    /* Load the special key to specify to use all global rules so
       we know how to initialize the structure */
    rules->all_global_rules = 0;
    if (global_rules != NULL) {
        LOAD_BOOL(SPAM, (NODEPS), ALL_GLOBAL_RULES, all_global_rules);
    }

    /* If the user specified to use all global rules, set the default values to those
     * in the globals, else set to the ones specified in the documentation
     * This only applies to boolean rules here.  The lists are also inherited later in the code.
     */
    if (rules->all_global_rules == 0) {
        retval |= config_spam_rule_init_defaults(rules);
    } else {
        retval |= config_spam_rule_init_defaults_from_globals(rules, global_rules);
    }

    /* Load the boolean variables */
    if (retval == 0) {
        retval |= config_spam_rule_load_variables(rules);
    }

    /* Load the white and black lists */
    if (retval == 0) {
        retval |= config_spam_rule_load_lists(rules, global_rules);
    }

    /* Load the white and black lists */
    /* Verify that the configuration is ok */
    if (retval == 0) {
        retval |= config_spam_rule_verify(rules);
    }

    return retval;
}

/* 
 * All the code after this will be automatically generated in the 
 * future from spam_rules.in
 */

/* Free all the strings from the ruleset */
int config_spam_rule_free(config_spam_rule_t * rules)
{
    if (rules->from_whitelist)
        STRLIST_FREE(rules->from_whitelist);
    if (rules->from_blacklist)
        STRLIST_FREE(rules->from_blacklist);
    if (rules->helo_whitelist)
        STRLIST_FREE(rules->helo_whitelist);
    if (rules->helo_blacklist)
        STRLIST_FREE(rules->helo_blacklist);
    if (rules->country_whitelist)
        STRLIST_FREE(rules->country_whitelist);
    if (rules->country_blacklist)
        STRLIST_FREE(rules->country_blacklist);
    if (rules->ip_whitelist)
        STRLIST_FREE(rules->ip_whitelist);
    if (rules->ip_blacklist)
        STRLIST_FREE(rules->ip_blacklist);

    return 0;
}

int config_spam_rule_init_defaults_from_globals(config_spam_rule_t * rules,
                                                config_spam_rule_t *
                                                global_rules)
{
    int retval = 0;

    rules->all_global_rules = global_rules->all_global_rules;
    rules->spam_check = global_rules->spam_check;
    rules->smtp_check = global_rules->smtp_check;
    rules->delivery_check = global_rules->delivery_check;
    rules->smtp_blocking = global_rules->smtp_blocking;
    rules->require_full_addr = global_rules->require_full_addr;
    rules->block_mail_from_self = global_rules->block_mail_from_self;
    rules->block_ip_in_addr = global_rules->block_ip_in_addr;
    rules->valid_from_domain = global_rules->valid_from_domain;
    rules->valid_bounce = global_rules->valid_bounce;
    rules->require_helo = global_rules->require_helo;
    rules->check_ip_reverse_dns = global_rules->check_ip_reverse_dns;

    retval |= STRLIST_NEW(rules->from_blacklist, emptystring);
    retval |= STRLIST_NEW(rules->from_whitelist, emptystring);
    retval |= STRLIST_NEW(rules->helo_blacklist, emptystring);
    retval |= STRLIST_NEW(rules->helo_whitelist, emptystring);
    retval |= STRLIST_NEW(rules->country_blacklist, emptystring);
    retval |= STRLIST_NEW(rules->country_whitelist, emptystring);
    retval |= STRLIST_NEW(rules->ip_blacklist, emptystring);
    retval |= STRLIST_NEW(rules->ip_whitelist, emptystring);

    return retval;
}

int config_spam_rule_init_defaults(config_spam_rule_t * rules)
{
    int retval = 0;

    rules->all_global_rules = SPAM_CONTROL_ALL_GLOBAL_RULES_DEFAULT;
    rules->spam_check = SPAM_CONTROL_SPAM_CHECK_DEFAULT;
    rules->virus_check = SPAM_CONTROL_VIRUS_CHECK_DEFAULT;
    rules->smtp_check = SPAM_CONTROL_SMTP_CHECK_DEFAULT;
    rules->delivery_check = SPAM_CONTROL_DELIVERY_CHECK_DEFAULT;
    rules->smtp_blocking = SPAM_CONTROL_SMTP_BLOCKING_DEFAULT;
    rules->require_full_addr = SPAM_CONTROL_REQUIRE_FULL_ADDR_DEFAULT;
    rules->block_mail_from_self = SPAM_CONTROL_BLOCK_MAIL_FROM_SELF_DEFAULT;
    rules->block_ip_in_addr = SPAM_CONTROL_BLOCK_IP_IN_ADDR_DEFAULT;
    rules->valid_from_domain = SPAM_CONTROL_VALID_FROM_DOMAIN_DEFAULT;
    rules->valid_bounce = SPAM_CONTROL_VALID_BOUNCE_DEFAULT;
    rules->require_helo = SPAM_CONTROL_REQUIRE_HELO_DEFAULT;
    rules->valid_helo_domain = SPAM_CONTROL_VALID_HELO_DOMAIN_DEFAULT;
    rules->mail_from_strict_addr_parse =
        SPAM_CONTROL_MAIL_FROM_STRICT_ADDR_PARSE_DEFAULT;
    rules->check_ip_reverse_dns = SPAM_CONTROL_CHECK_IP_REVERSE_DNS_DEFAULT;

    retval |= STRLIST_NEW(rules->from_blacklist, emptystring);
    retval |= STRLIST_NEW(rules->from_whitelist, emptystring);
    retval |= STRLIST_NEW(rules->helo_blacklist, emptystring);
    retval |= STRLIST_NEW(rules->helo_whitelist, emptystring);
    retval |= STRLIST_NEW(rules->country_blacklist, emptystring);
    retval |= STRLIST_NEW(rules->country_whitelist, emptystring);
    retval |= STRLIST_NEW(rules->ip_blacklist, emptystring);
    retval |= STRLIST_NEW(rules->ip_whitelist, emptystring);

    return retval;
}

int config_spam_rule_load_variables(config_spam_rule_t * rules)
{
    int retval = 0;

    syslog(LOG_DEBUG, "XXX LOADING VARIABLES");

    LOAD_BOOL(SPAM, (NODEPS), ALL_GLOBAL_RULES, all_global_rules);
    LOAD_BOOL(SPAM, (NODEPS), SPAM_CHECK, spam_check);
    LOAD_BOOL(SPAM, (NODEPS), VIRUS_CHECK, virus_check);
    LOAD_BOOL(SPAM, (rules->spam_check), SMTP_CHECK, smtp_check);
    LOAD_BOOL(SPAM, (rules->spam_check), DELIVERY_CHECK, delivery_check);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check), SMTP_BLOCKING,
              smtp_blocking);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check), REQUIRE_FULL_ADDR,
              require_full_addr);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check),
              BLOCK_MAIL_FROM_SELF, block_mail_from_self);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check), BLOCK_IP_IN_ADDR,
              block_ip_in_addr);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check), VALID_FROM_DOMAIN,
              valid_from_domain);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check), VALID_BOUNCE,
              valid_bounce);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check), REQUIRE_HELO,
              require_helo);
    LOAD_BOOL(SPAM,
              (rules->spam_check && rules->smtp_check
               && rules->require_helo), VALID_HELO_DOMAIN, valid_helo_domain);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check),
              MAIL_FROM_STRICT_ADDR_PARSE, mail_from_strict_addr_parse);
    LOAD_BOOL(SPAM, (rules->spam_check && rules->smtp_check),
              CHECK_IP_REVERSE_DNS, check_ip_reverse_dns);

    return retval;
}

int config_spam_rule_load_lists(config_spam_rule_t * rules,
                                config_spam_rule_t * global_rules)
{
    int retval = 0;

    LOAD_LIST(SPAM,
              (rules->spam_check && rules->smtp_check
               && rules->valid_from_domain), FROM_BLACKLIST, from_blacklist);
    LOAD_LIST(SPAM,
              (rules->spam_check && rules->smtp_check
               && rules->valid_from_domain), FROM_WHITELIST, from_whitelist);
    LOAD_LIST(SPAM, (rules->spam_check && rules->smtp_check), HELO_WHITELIST,
              helo_whitelist);
    LOAD_LIST(SPAM, (rules->spam_check && rules->smtp_check), HELO_BLACKLIST,
              helo_blacklist);
    LOAD_LIST(SPAM,
              (rules->spam_check && rules->smtp_check
               && rules->valid_from_domain), COUNTRY_WHITELIST,
              country_whitelist);
    LOAD_LIST(SPAM,
              (rules->spam_check && rules->smtp_check
               && rules->valid_from_domain), COUNTRY_BLACKLIST,
              country_blacklist);
    LOAD_LIST(SPAM, (rules->spam_check && rules->smtp_check), IP_WHITELIST,
              ip_whitelist);
    LOAD_LIST(SPAM, (rules->spam_check && rules->smtp_check), IP_BLACKLIST,
              ip_blacklist);

    return retval;
}


static int config_spam_rule_load_globals(void)
{
    static int loaded = 0;

    if (loaded == 1) {
        syslog(LOG_DEBUG, "Not loading Globals, because they're already loaded");
        return (0);
    }
    loaded = 1;

#ifdef USE_LM_DATABASE
    if (config.use_database != 0) {
        /* Open the database connection */
        if (db_init() != -1) {

            /* TODO Change this to the global rule name */
            LM_STRING_NEW(dbwhere, "and id=0");
            if (LM_CONFIG_DB_SET_PARAM
                (db, config.spam_table, dbselectcol, dbkeycol, dbwhere)) {
                SYSLOG((LOG_ERR, "error setting up spam db paramters."));
            }

            /* Load the global spam rules from the control file directory */
            config_spam_rule_read(NULL, &global_spam_rules);
            LM_CONFIG_CLOSE;
            LM_STRING_FREE(dbwhere);

            syslog(LOG_DEBUG, "Loaded Globals from Database");
            return (0);
        }  else {
            syslog(LOG_DEBUG, "DB initialization failed");
        }

        /* Fall through to next method if can't open Database */
    }

#endif


    /* XXX: can't load globals in ext prog version yet */


#ifdef USE_LM_DBFILE

    if (config.use_dbfile) {
        load_global_spam_rules_dbfile(&global_spam_rules);

        syslog(LOG_DEBUG, "Loaded Globals from DBfile");
        return (0);
    }

#endif

    syslog(LOG_DEBUG, "No globals to load");
    return (0);
}
