/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * Copyright (c) 2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Josh Wilsdon <josh@wizard.ca>
 *
 * Version: $Id: load_controls.c,v 1.3 2003/09/24 23:45:22 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 *
 */

/* standard libraries */
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <syslog.h>
#include <unistd.h>
#define _GNU_SOURCE             /* do this so we can use strdup() */
#include <string.h>
#include <assert.h>
#include <errno.h>

/* local libraries */
#include "controls.h"
#include "load_controls.h"


/* NOTE:
 * 
 * config and config_type_info are assumed to be defined in the calling
 * module and available here as a global.  They are expected to have the
 * proper data in them.
 *
 * ALSO: 
 *
 * This module was hastily written, and should be refactored at a later
 * date, when there is sufficient time.
 * 
 */


extern config_t config;
extern config_type_t config_type_info[];

/* load controls (PRIVATE) */
static int loadControlBool(config_type_t *ptr);
static int loadControlInt(config_type_t *ptr);
static int loadControlString(config_type_t *ptr);
static int loadControlProg(config_type_t *ptr);
static int loadControlDir(config_type_t *ptr);
static int loadControlFile(config_type_t *ptr);
static void loadError(char *name, char *reason, int def);
static void loadInfoInit(void);
static void loadInfoStr(char *name, char *value, int def);
static void loadInfoInt(char *name, int value, int def);


/* read controls (PRIVATE) */
static int readControl(char *control, char **value, int nlines);
static int readControlBool(char *control, int *out);
static int readControlInt(char *control, int *out);
static int readControlString(char *control, char **out);
static int readControlProg(char *control, char **out);
static int readControlDir(char *control, char **out);
static int readControlFile(char *control, char **out);
static char *readControlErrStr(int error);

/* misc helpers (PRIVATE) */
static int enterControlDir(void);
static int appendString(char **dest, int *destbytes, char *src);
static int convertStrToBool(char *str);

/* global data for this module */
int control_load_verbosity = 0;

#ifdef TEST_MAIN
int main()
{
    int ret;

    if ((ret = loadControls(1)) != 0) {
        printf("Error loading controls: %d\n", ret);
    }

    exit(0);
}
#endif

int enterControlDir(void)
{
    char *controldir;
    struct stat statinfo;

    /* 
     * check for environment variable MAGICMAILCONTROLDIR which if set, we will
     * use as directory to read the control files from
     */
    controldir = getenv("MAGICMAILCONTROLDIR");
    if (controldir != NULL) {
        if ((stat(controldir, &statinfo) == 0) && (S_ISDIR(statinfo.st_mode))) {
            /* exists and is directory, so use it to read controls */
        } else {
            /* wanted different control dir, but it was invalid */
            if (control_load_verbosity > 9) {
                printf("WARNING: MAGICMAILCONTROLDIR invalid, using default\n");
            }
            controldir = NULL;
        }
    }

    /* if it wasn't set yet, use default */
    if (controldir == NULL) {
        controldir = DEFAULT_MAGICMAILCONTROLDIR;
    }

    /* switch to controldir so we can just read the files */
    if (chdir(controldir) != 0) {
        if (control_load_verbosity > 9) {
            printf("error loading controls from directory[%s]: %s\n", controldir, strerror(errno));
        }
        return (-1);
    } else {
        if (control_load_verbosity > 9) {
            printf("DEBUG: loading controls from directory: %s\n", controldir);
        }
    }

    return (0);
}

int loadControls(int verbosity)
{
    config_type_t *ptr;

    control_load_verbosity = verbosity; /* set the verbosity for this module */

    /* start with empty config */
    bzero(&config, sizeof(config));

    if (enterControlDir() != 0) {
        return (-1);
    }

    /* from this point, control files can be read in the current dir */

    loadInfoInit();

    ptr = config_type_info;
    while (ptr->name != NULL) {
        switch(ptr->type) {
            case CONTROL_BOOL:
                loadControlBool(ptr);
                break;
            case CONTROL_INT:
                loadControlInt(ptr);
                break;
            case CONTROL_STRING:
                loadControlString(ptr);
                break;
            case CONTROL_PROG:
                loadControlProg(ptr);
                break;
            case CONTROL_DIR:
                loadControlDir(ptr);
                break;
            case CONTROL_FILE:
                loadControlFile(ptr);
                break;
            default:
                /* corrupted common/control_files file */
                break;
        }
        ptr++;
    }

    return(0);
}

int loadControlInt(config_type_t *ptr)
{
    int *valueptr, ret;
    long tmpout;

    assert(ptr != NULL);
    valueptr = (int *)((unsigned int)&config + ptr->offset);

    if ((ret = readControlInt(ptr->name, valueptr)) != 0) {
        loadError(ptr->name, readControlErrStr(ret), 0);
        if (ptr->defvalue == NULL) {
            /* no default value */
            loadError(ptr->name, "No default value", 1);
            return (-6);
        }

        errno = 0;
        tmpout = strtol(ptr->defvalue, (char **)NULL, 10);
        if ((errno == ERANGE) || (tmpout > INT_MAX)) {
            loadError(ptr->name, readControlErrStr(-3), 1);
            return (-3);    /* overflow */
        }
        loadInfoInt(ptr->name, tmpout, 1);
        *valueptr = (int)tmpout;

        return(0);
    } else {
        loadInfoInt(ptr->name, *valueptr, 0);
    }

    return(0);
}

int loadControlBool(config_type_t *ptr)
{
    int *valueptr, ret;

    assert(ptr != NULL);
    valueptr = (int *)((unsigned int)&config + ptr->offset);

    if ((ret = readControlBool(ptr->name, valueptr)) != 0) {
        loadError(ptr->name, readControlErrStr(ret), 0);
        if ((ret = convertStrToBool(ptr->defvalue)) == -1) {
            loadError(ptr->name, readControlErrStr(ret), 1);
            return (-4);
        }
        *valueptr = (int)ret;
        loadInfoInt(ptr->name, *valueptr, 1);
    } else {
        loadInfoInt(ptr->name, *valueptr, 0);
    }

    return(0);
}

int loadControlString(config_type_t *ptr)
{
    char **valueptr;
    int ret;

    assert(ptr != NULL);
    valueptr = (char **)((unsigned int)&config + ptr->offset);

    if ((ret = readControlString(ptr->name, valueptr)) != 0) {
        loadError(ptr->name, readControlErrStr(ret), 0);
        memcpy(valueptr, &(ptr->defvalue), sizeof(valueptr));
        loadInfoStr(ptr->name, *valueptr, 1);
        return(ret);
    } else {
        loadInfoStr(ptr->name, *valueptr, 0);
    }

    return(0);
}

int loadControlProg(config_type_t *ptr)
{
    char **valueptr, *tmpptr;
    struct stat statinfo;
    int ret;

    assert(ptr != NULL);
    valueptr = (char **)((unsigned int)&config + ptr->offset);

    /* XXX: special case: check for QMAILQUEUE environment variable
     *      if it exists set control.qmail_queue to this
     */
    if (strncmp(ptr->name, "qmail_queue", 11) == 0) {
        if ((tmpptr = getenv("QMAILQUEUE")) != NULL) {
            if ((stat(tmpptr, &statinfo) == 0) && (S_ISREG(statinfo.st_mode) != 0)) {
                memcpy(valueptr, &tmpptr, sizeof(valueptr));
                loadInfoStr(ptr->name, *valueptr, 2);
                return(0);
            }
        }
    }

    if ((ret = readControlProg(ptr->name, valueptr)) != 0) {
        loadError(ptr->name, readControlErrStr(ret), 0);
        memcpy(valueptr, &(ptr->defvalue), sizeof(valueptr));
        loadInfoStr(ptr->name, *valueptr, 1);
        return(ret);
    } else {
        loadInfoStr(ptr->name, *valueptr, 0);
    }

    return(0);
}

int loadControlDir(config_type_t *ptr)
{
    char **valueptr;
    int ret;

    assert(ptr != NULL);
    valueptr = (char **)((unsigned int)&config + ptr->offset);

    if ((ret = readControlDir(ptr->name, valueptr)) != 0) {
        loadError(ptr->name, readControlErrStr(ret), 0);
        memcpy(valueptr, &(ptr->defvalue), sizeof(valueptr));
        loadInfoStr(ptr->name, *valueptr, 1);
        return(ret);
    } else {
        loadInfoStr(ptr->name, *valueptr, 0);
    }

    return(0);
}

int loadControlFile(config_type_t *ptr)
{
    char **valueptr;
    int ret;

    assert(ptr != NULL);
    valueptr = (char **)((unsigned int)&config + ptr->offset);

    if ((ret = readControlFile(ptr->name, valueptr)) != 0) {
        loadError(ptr->name, readControlErrStr(ret), 0);
        memcpy(valueptr, &(ptr->defvalue), sizeof(valueptr));
        loadInfoStr(ptr->name, *valueptr, 1);
        return(ret);
    } else {
        loadInfoStr(ptr->name, *valueptr, 0);
    }

    return(0);
}

static void loadError(char *name, char *reason, int def)
{
    if (control_load_verbosity > 10) {
        if (def == 0) {
            printf("error loading from %s: %s\n", name, reason);
        } else {
            printf("error loading default for %s: %s\n", name, reason);
        }
    }
    return;
}

static void loadInfoInit(void)
{
    if (control_load_verbosity > 9) {
        printf(" where\t| option\t\t\t| value\n");
        printf("------------------------------------------------------------\n");
    }

    return;
}

static void loadInfoStr(char *name, char *value, int def)
{
    char *how;
    char tabs[10];
    int count, i, strwidth, width = 32;

    if (def == 0) {
        how = "loaded";
    } else if (def == 2) {
        how = "environ";
    } else {
        how = "default";
    }

    bzero(tabs, sizeof(tabs)); 
    width = (width / 8) * 8;
    strwidth = strlen(name) + 2;

    count = ((width - strwidth) / 8);
    if (((width - strwidth) % 8)) {
         count++;
    }
    for(i=0; i< count; i++) {
        tabs[i] = '\t';
    }
    tabs[i] = '\0';

    if (control_load_verbosity > 9) {
        printf("%s\t| %s%s| %s\n", how, name, tabs, value);
    }

    return;
}

static void loadInfoInt(char *name, int value, int def)
{
    char strbuf[1024], *strptr;

    if (snprintf(strbuf, sizeof(strbuf), "%d", value) > 0) {
        strptr = strdup(strbuf);
        if (strptr != NULL) {
            loadInfoStr(name, strptr, def);
            free(strptr);
        }
    }
    
    return;
}


/* Returns:
 *
 *  -1 could not open file
 *  -2 memory allocation failure
 * >=0 number of lines read
 *
 * Side Effects:
 *
 *  value must be free()'d by caller
 *
 */
int readControl(char *control, char **value, int nlines)
{
    int ignore, nread, outsize;
    unsigned int i;
    char *outbuf, chunkbuf[1024];
    FILE *fp;

    assert(control != NULL);
    assert(value != NULL);

    fp = fopen(control, "r");
    if (fp == NULL) {
        return (-1);
    }

    outbuf = NULL;  
    outsize = 0;    /* no memory yet allocated */
    nread = 0;      /* have not yet read any lines */
    while(((nlines == 0) || (nread < nlines)) 
            && (fgets(chunkbuf, sizeof(chunkbuf), fp) != NULL)) {
        ignore = 0;
        for (i=0; i < strlen(chunkbuf); i++) {
            if (chunkbuf[i] == '#') {
                ignore = 1;     /* comment */
                break;
            }
            if (!isspace(chunkbuf[i])) {
                ignore = 0;     /* don't ignore */
                break;
            }
        }
        if (i == strlen(chunkbuf)) {
            ignore = 1;         /* no non-whitespace chars found in line */
        }
        if (ignore == 0) {
            nread++;
            if (appendString(&outbuf, &outsize, chunkbuf) != 0) {
                fclose(fp);
                return (-2);
            }
        }
    }

    fclose(fp);
    memcpy(value, &outbuf, sizeof(value));

    return (nread);     /* number of lines read successfully */
}


 
/*
 * RETURN VALUES: 
 *
 *  -4   bad value
 *
 */
int readControlBool(char *control, int *out)
{
    char *value;
    int ret;
    long tmpout = 3;        /* initialize to bad value so we notice problems */

    value = NULL;
    if ((ret = readControl(control, &value, 1)) < 0) {
        return (ret);
    }

    assert(value != NULL);
    /* from here, value has memory allocated and needs to be free()'d */

    if (value[strlen(value) - 1] == '\n') {
        value[strlen(value) - 1] = '\0';
    }

    if ((tmpout = convertStrToBool(value)) == -1) {
        free(value);
        return (-4);
    }

    *out = (int) tmpout;
    free(value);
    return (0);
}


/*
 * RETURN VALUES: 
 *
 *  -3   overflow
 *
 */
int readControlInt(char *control, int *out)
{
    char *value;
    int ret;
    long tmpout;

    value = NULL;
    if ((ret = readControl(control, &value, 1)) < 0) {
        return (ret);
    }

    assert(value != NULL);
    /* from here, value has memory allocated and needs to be free()'d */

    errno = 0;
    tmpout = strtol(value, (char **)NULL, 10);
    if ((errno == ERANGE) || (tmpout > INT_MAX)) {
        free(value);
        return (-3);    /* overflow */
    }

    *out = (int) tmpout;
    free(value);
    return (0);
}

int readControlString(char *control, char **out)
{
    int ret;
    char *value;

    if ((ret = readControl(control, out, 1)) < 0) {
        return (ret);
    }

    value = *out;
    if (value[strlen(value) - 1] == '\n') {
        value[strlen(value) - 1] = '\0';
    }

    /* on success, memory is allocated that needs to be free()'d */

    return 0;
}

int readControlFile(char *control, char **out)
{
    char *value;
    int ret;
    struct stat statinfo;

    value = NULL;
    if ((ret = readControl(control, &value, 1)) < 0) {
        return (ret);
    }

    assert(value != NULL);
    /* from here, value has memory allocated and needs to be free()'d */

    if (value[strlen(value) - 1] == '\n') {
        value[strlen(value) - 1] = '\0';
    }

    if ((stat(value, &statinfo) != 0) || (S_ISREG(statinfo.st_mode) == 0)) {
        /* not a file */
        free(value);
        return (-1);
    }

    memcpy(out, &value, sizeof(out));

    return (0);
}


int readControlDir(char *control, char **out)
{
    char *value;
    int ret;
    struct stat statinfo;

    value = NULL;
    if ((ret = readControl(control, &value, 1)) < 0) {
        return (ret);
    }

    assert(value != NULL);
    /* from here, value has memory allocated and needs to be free()'d */

    if (value[strlen(value) - 1] == '\n') {
        value[strlen(value) - 1] = '\0';
    }

    if ((stat(value, &statinfo) != 0) || (S_ISDIR(statinfo.st_mode) == 0)) {
        /* not a directory */
        free(value);
        return (-1);
    }

    memcpy(out, &value, sizeof(out));
    return (0);
}

int readControlProg(char *control, char **out)
{
    char *value;
    int ret;
    struct stat statinfo;

    value = NULL;
    if ((ret = readControl(control, &value, 1)) < 0) {
        return (ret);
    }

    assert(value != NULL);
    /* from here, value has memory allocated and needs to be free()'d */

    if (value[strlen(value) - 1] == '\n') {
        value[strlen(value) - 1] = '\0';
    }

    if ((stat(value, &statinfo) != 0) || (S_ISREG(statinfo.st_mode) == 0)) {
        /* not a regular file  */
        /* TODO: should also check to make sure we can execute it */
        free(value);
        return (-1);
    }

    memcpy(out, &value, sizeof(out));
    return (0);
}

char *readControlErrStr(int error)
{
    switch(error) {
        case -1:
            return "File I/O Error";
        case -2:
            return "Memory Allocation Failure";
        case -3:
            return "Overflow";
        case -4:
            return "Invalid Value";
        default:
            return "Unknown Error";
    }
}

int appendString(char **dest, int *destbytes, char *src)
{
    int newsize;
    char *newptr;
    int init = 0;

    assert(destbytes != NULL);
    assert(src != NULL);

    if ((*dest == NULL) || (*destbytes == 0)) {
        init = 1;
        if ((strlen(src) + 1) < 1024) {
            newsize = 1024;
        } else {
            newsize = strlen(src) + 1;
        }
    } else {
        newsize = *destbytes;
    }

    if (*dest != NULL) {
        while ((strlen(src) + strlen(*dest) + 1) >= (size_t) newsize) {
            newsize *= 2; 
        }
    }

    newptr = (char *)realloc(*dest, (size_t)newsize);
    if (newptr == NULL) {
        return (-1);
    }

    if (init == 1) {
        newptr[0] = '\0';
    }
    strncat(newptr, src, strlen(src));
    memcpy(dest, &newptr, sizeof(dest));
    *destbytes = newsize;

    return (0);
}

int convertStrToBool(char *str)
{
    assert(str != NULL);

    if ((strcasecmp(str, "true") == 0)
            || (strcasecmp(str, "on") == 0)
            || (strcasecmp(str, "yes") == 0)
            || (strcmp(str, "1") == 0)) {
        return (1);
    } else if ((strcasecmp(str, "false") == 0)
                    || (strcasecmp(str, "off") == 0)
                    || (strcasecmp(str, "no") == 0)
                    || (strcmp(str, "0") == 0)) {
        return (0);
    } 
    
    return (-1);
}
