/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *
 * CVS Id: $Id: check_user.c,v 1.41 2003/10/27 20:41:02 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

#include <ctype.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <errno.h>


/* lmlib includes */
#include <liblm.h>

/* Application includes */
#include "check_user.h"
#include "smtp.h"
#include "cdb.h"

#include "common/controls.h"
#include "common/load_controls.h"

extern config_t config;
extern config_type_t config_type_info[];

int msd_check_rcpt_user_db(lm_string_t * addr);
int msd_check_rcpt_user_dbfile(lm_string_t * addr);
static int check_user_ext_prog(char *email, lm_string_t *spamdir);


int msd_check_rcpt_user(lm_string_t * paddr, lm_string_t * spamdir)
{
    char *buffer;
    char postmaster[] = "postmaster";
    char postmasterat[] = "postmaster@";
    int ret = -1;

    /* look for postmaster address */
    buffer = LM_STRING_BUFFER(*paddr);
    if (strlen(buffer) == strlen(postmaster)) {
        if (strncasecmp(postmaster, buffer, strlen(postmaster)) == 0) {
            return (0);
        }
    } else if (strncasecmp(postmasterat, buffer, strlen(postmasterat)) == 0) {
        return (0);
    }

    /* not to postmaster or postmaster@ */

#ifdef USE_LM_DATABASE

    if (config.use_database == 1) {
        if ((ret = msd_check_rcpt_user_db(paddr)) != -1) {
            /* did not have an error so it did it's job */
            return (ret);
        }

        /* database check failed, so fall through to next method */
    }

#endif 

    /* CHECK USING EXTERNAL PROGRAM IF IT IS SET */

    if (config.ext_check_user_prog != NULL) {
        if ((ret = check_user_ext_prog(LM_STRING_BUFFER(*paddr), spamdir)) != -1) {
            /* check_user program did not have an error so it did it's job */
            return (ret);
        } 

        /* check_user program failed, so fall through to normal method */
    }

#ifdef USE_LM_DBFILE

    if (config.use_dbfile == 1) {
        if ((ret = msd_check_rcpt_user_dbfile(paddr)) != -1) {
            /* did not have an error so it did it's job */
            return (ret);
        }

        /* check failed, so fall through to next method */
    }

#endif

    return (ret);
}


int check_user_ext_prog(char *email, lm_string_t* spamdir)
{
    int retval, wpret;
    char childret;
    int childpid, bufidx;
    int childstdout[2];
    char result_buffer[1024];
    lm_string_t outstring = NULL;

    char *prog;
    char *args[] = {
        NULL, NULL, NULL
    };

    /* Default return value for this function is user valid, as
     * If no program is specified, we already know we matched a -default
     * file (or else this function wouldn't have been called).  This function
     * just provides an extra level for virtual domain hosters to check a
     * database or whatever they want.
     */
    retval = 0;

    if (pipe(childstdout) != 0) {
        syslog(LOG_ERR, "external check_user program pipe error: %s", strerror(errno));
        return (-1);
    }

    switch (childpid = fork()) {
        case 0:
            /* Launch the program in the child process */
            prog = config.ext_check_user_prog;
            args[0] = prog;
            args[1] = email;

            close(childstdout[0]);
            if (dup2(childstdout[1], 1) == -1) {
                syslog(LOG_ERR, "external check_user program dup2(): %s", strerror(errno));
                _exit(-1);
            }

            syslog(LOG_DEBUG, "running external check user program: [%s] [%s]\n", prog, email);

            if (execvp(prog, args)) {
                syslog(LOG_ERR, "external check_user program exec error: %s", strerror(errno));
            }

            _exit(-1);
            break;

        case -1:
            syslog(LOG_ERR, "external check_user program fork error: %s", strerror(errno));
            return(-1);
    }

    /* PARENT */

    close(childstdout[1]);

    /* XXX output longer than sizeof(result_buffer) will be truncated */
    for (bufidx = 0; (bufidx < (sizeof(result_buffer) - 1)); bufidx++) {
        /* break on eof, error or '\n' */
        if ((read(childstdout[0], &result_buffer[bufidx], 1) != 1)
            || (result_buffer[bufidx] == '\n')) {
            break;      /* done, leave for loop */
        }
    }
    result_buffer[bufidx] = '\0';

    syslog(LOG_DEBUG, "result_buffer[%s]", result_buffer);

    if (strlen(result_buffer) > 0) {
        /* copy to the out variable */
        printf("OUTSTRING: %p\n", outstring);
        LM_STRING_NEW(outstring, result_buffer);
        if (outstring != NULL) { 
            printf("OUTSTRING: %p\n", outstring);
            *spamdir = outstring;
        }
    }

    /* Wait for the child to finish */
    do {
        wpret = waitpid(childpid, &retval, 0);
    } while ((wpret == -1) && (errno == EINTR));

    if (wpret != childpid) {
        /* Something bad happened */
        return -1;
    }

    /* Doing this to be safe.  WEXTISTATUS returns 0-255, which won't
     * do if our return value is an int.  Just be sure that it gets
     * sign extended by saving the exit status to a char and then
     * implicitly casting it to an int with the next assign
     */
    childret = WEXITSTATUS(retval);
    retval = childret;

    return retval;
}
