/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/FSCL.txt
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Josh Wilsdon <josh@wizard.ca>
 *
 * CVS Id: $Id: main.c,v 1.7 2003/09/19 18:56:03 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

#include <dlfcn.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <db.h>

#include "include/magicdb.h"

void writeRecord(int type, int keysize, char *key, int datasize, char *data);
int openDB(int type);
void closeDB(void);
int createDB(void);
int verifyDB(const char *dbname);
DB *dbp;

int main()
{
    void *handle;
    char *error;
    int (*init) (void);
    int (*cleanup) (void);
    int (*foreachRecord) (int type,
                          void (*func) (int type, int keysize, char *key,
                                        int datasize, char *data));
    int ret;

    handle = dlopen("/usr/local/modules/mod_postgresql.so", RTLD_LAZY);
    if (handle == NULL) {
        fprintf(stderr, "dlopen(): %s\n", dlerror());
        exit(1);
    }

    init = dlsym(handle, "init");
    if ((error = dlerror()) != NULL) {
        fprintf(stderr, "dlsym(): %s\n", error);
        exit(1);
    }
    init();

    foreachRecord = dlsym(handle, "foreachRecord");
    if ((error = dlerror()) != NULL) {
        fprintf(stderr, "dlsym(): %s\n", error);
        exit(1);
    }

    if (openDB(SPAM_RULES) != 0) {
        fprintf(stderr, "db_open failed\n");
        exit(1);
    }
    if (foreachRecord(SPAM_RULES, writeRecord) != 0) {
        fprintf(stderr, "foreachRecord() failed\n");
        exit(1);
    }
    // printf("countPairs(%s): %d\n", SPAM_DATABASE, countPairs());
    closeDB();

    if (openDB(USER_INFO) != 0) {
        fprintf(stderr, "db_open failed\n");
        exit(1);
    }
    if (foreachRecord(USER_INFO, writeRecord) != 0) {
        fprintf(stderr, "foreachRecord() failed\n");
        exit(1);
    }
    // printf("countPairs(%s): %d\n", USER_DATABASE, countPairs());
    closeDB();

    if (createDB() == 0) {
        if ((ret = verifyDB(USER_DATABASE)) != 0) {
            fprintf(stderr, "verification of database [%s] failed\n",
                    USER_DATABASE);
            exit(1);
        }
        if ((ret = verifyDB(SPAM_DATABASE)) != 0) {
            fprintf(stderr, "verification of database [%s] failed\n",
                    SPAM_DATABASE);
            exit(1);
        }
    }

    cleanup = dlsym(handle, "cleanup");
    if ((error = dlerror()) != NULL) {
        fprintf(stderr, "dlsym(): %s\n", error);
        exit(1);
    }
    cleanup();

    dlclose(handle);

    exit(0);
}

/* 
 * According to the documentation, DB->close() can fail to sync or on other
 * catastrophic conditions.  If it fails, we cannot be guaranteed the state
 * of the database, so we exit(1) to fail the whole run.
 *
 */
void closeDB(void)
{
    if (dbp->close(dbp, 0) != 0) {
        fprintf(stderr, "db_close failed\n");
        exit(1);
    }
}

int countPairs(void)
{
    DB_HASH_STAT *stats;

    if (dbp->stat(dbp, &stats, NULL, 0) == 0) {
        return ((int) stats->hash_ndata);
    }

    return (-1);
}

int verifyDB(const char *dbname)
{
    if (dbname == NULL) {
        return (1);
    }

    return (dbp->verify(dbp, dbname, NULL, NULL, 0));
}

int createDB(void)
{
    int ret;

    if ((ret = db_create(&dbp, NULL, 0)) != 0) {
        fprintf(stderr, "db_create: %s\n", db_strerror(ret));
        return (1);
    }

    return (0);
}

int openDB(int type)
{
    char *filename;
    int ret;

    switch (type) {
        case SPAM_RULES:
            filename = SPAM_DATABASE;
            break;
        case USER_INFO:
            filename = USER_DATABASE;
            break;
        default:
            fprintf(stderr, "openDB: unknown type: %d\n", type);
            return (1);
            break;
    }

    /* Create the database handle. */
    if (createDB() == 0) {

        /* Open the underlying database. */
        if ((ret =
             dbp->open(dbp, filename, NULL, DB_HASH, DB_CREATE, 0664)) == 0) {
            return (0);
        } else {
            dbp->err(dbp, ret, "%s", filename);
        }
    }

    return (1);
}

void writeRecord(int type, int keysize, char *key, int datasize, char *data)
{
    DBT dbkey, dbdata;
    int ret;

    type = type;                /* remove gcc warning */

    memset(&dbkey, 0, sizeof(dbkey));
    memset(&dbdata, 0, sizeof(dbdata));
    dbkey.data = key;
    dbkey.size = keysize;
    dbdata.data = data;
    dbdata.size = datasize;

    /* Store a key/data pair. */
    if ((ret = dbp->put(dbp, NULL, &dbkey, &dbdata, 0)) != 0) {
        dbp->err(dbp, ret, "DB->put");
        exit(1);
    }

    return;
}
