/*
 * COPYRIGHT INFORMATION - DO NOT REMOVE
 * "Portions Copyright (c) 2002-2003 LinuxMagic Inc. All Rights Reserved.
 *
 * This file contains Original Code and/or Modifications of Original Code as
 * defined in and that are subject to the Free Source Code License Version
 * 1.0 (the 'License'). You may not use this file except in compliance with
 * the License. Please obtain a copy of the License at:
 *
 * http://www.linuxmagic.com/opensource/licensing/
 *
 * and read it before using this file.
 *
 * The Original Code and all software distributed under the License are
 * distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND LINUXMAGIC HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT. Please see
 * the License for the specific language governing rights and limitations
 * under the License."
 *
 * Please read the terms of this license carefully. By using or downloading
 * this software or file, you are accepting and agreeing to the terms of this
 * license with LinuxMagic Inc. If you are agreeing to this license on behalf
 * of a company, you represent that you are authorized to bind the company to
 * such a license. If you do not meet this criterion or you do not agree to
 * any of the terms of this license, do NOT download, distribute, use or alter
 * this software or file in any way.
 *
 * Author(s): Burton Samograd <burton@wizard.ca>
 *            Michael Peddemors <michael@wizard.ca>
 *            Josh Wilsdon <josh@wizard.ca>
 *
 * CVS Id: $Id: auth_chkp.c,v 1.8 2003/10/22 19:04:52 josh Exp $
 *
 * DO NOT MODIFY WITHOUT CONSULTING THE LICENSE
 */

/* C includes */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>
#include <sys/stat.h>
#include <pwd.h>

#include <liblm.h>

#include "config_msd.h"
#include "smtp.h"
#include "auth.h"
#include "auth.h"
#include "auth_chkp.h"
#include "config_qmail.h"
#include "magic-smtpd.h"

#include "common/controls.h"
#include "common/load_controls.h"

extern config_t config;
extern config_type_t config_type_info[];


static int checkpassword(char *user, char *pass);

/* 
 * SUMMARY
 *
 *   This runs the external checkpassword program with an argument
 *   of /bin/true and checks the return value:
 *
 * RETURN VALUES
 *
 *   0 success
 *   1 failure
 *  -1 error 
 *
 * SIDE EFFECTS
 *
 *   Will write messages to syslog on error.
 *   Will attempt to fork() and child will attempt to exec() a subprocess.
 *
 */
int checkpassword(char *user, char *pass)
{
    pid_t child;
    int childpipe[2];
    int wstatus;
    char extra[] = "user/pass";
    int userlen, passlen, extralen;
    char *program;

    if ((config.ext_check_passwd_prog == NULL) || (strlen(config.ext_check_passwd_prog) == 0)) {
        return (-1);
    }
    program = config.ext_check_passwd_prog;

    /* these lengths include the terminating \0 */
    userlen = strlen(user) + 1;
    passlen = strlen(pass) + 1;
    extralen = strlen(extra) + 1;

    /* make sure we don't have too much data for checkpassword */
    if ((userlen + passlen + extralen) > MAX_EXT_PROG_DATA) {
        return (-1);
    }

    close(3);
    if (pipe(childpipe) == -1) {
        syslog(LOG_ERR,
               "AUTH ERROR: auth_chkp: checkpassword(): pipe() failed");
        return (-1);
    }
    if (childpipe[0] != 3) {
        /* how did we not get descriptor 3 ? */
        syslog(LOG_ERR,
               "AUTH ERROR: auth_chkp: checkpassword(): pipe descriptor is not 3");
        return (-1);
    }

    switch (child = fork()) {
        case -1:
            /* error */
            syslog(LOG_ERR,
                   "AUTH ERROR: auth_chkp: checkpassword(): fork() failed");
            break;
        case 0:
            /* child */
            close(childpipe[1]);    /* close the write end */
            execl(program, program, EXT_TRUE_PROG, NULL);
            syslog(LOG_ERR,
                   "AUTH ERROR: auth_chkp: checkpassword(): execl() failed");
            _exit(1);
        default:
            /* parent */
            close(childpipe[0]);    /* close the read end */
            if (write(childpipe[1], user, userlen) != userlen) {
                syslog(LOG_ERR,
                       "AUTH ERROR: auth_chkp: checkpassword(): write(user): short write");
                return (-1);
            }
            if (write(childpipe[1], pass, passlen) != passlen) {
                syslog(LOG_ERR,
                       "AUTH ERROR: auth_chkp: checkpassword(): write(pass): short write");
                return (-1);
            }
            if (write(childpipe[1], extra, extralen) != extralen) {
                syslog(LOG_ERR,
                       "AUTH ERROR: auth_chkp: checkpassword(): write(extra): short write");
                return (-1);
            }
            close(childpipe[1]);    /* finished writing now too */

            if (waitpid(child, &wstatus, 0) == -1) {
                syslog(LOG_ERR,
                       "AUTH ERROR: auth_chkp: checkpassword(): waitpid() failed");
                return (-1);
            }
            if (WIFEXITED(wstatus) == 0) {
                syslog(LOG_ERR,
                       "AUTH ERROR: auth_chkp: checkpassword(): child crashed");
                return (-1);
            }
            if (WEXITSTATUS(wstatus) == 0) {
                /* success */
                return (0);
            }

            /* auth failed */
            return (1);
            break;
    }

    /* error */
    return (-1);
}


/* 
 * SUMMARY
 *
 *   Convert lm_string_t's to C strings and send them on to the checkpassword
 *   program for now, since the only type supported is LOGIN. 
 *
 * RETURN VALUES
 *
 *   LM_AUTH_UNSUPPORTED    unsupported auth type
 *   LM_AUTH_BAD_DATA       invalid input
 *   LM_AUTH_ERROR          error in checkpassword
 *   LM_AUTH_FAILED         authentication failed
 *   LM_AUTH_OK             successful authentication
 *
 * SIDE EFFECTS
 *
 *   Will call the checkpassword() function.
 *
 */
int msd_do_auth_chkp(int type, lm_string_t username, lm_string_t password)
{
    char *user, *pass;
    int retval;

    user = LM_STRING_BUFFER(username);
    pass = LM_STRING_BUFFER(password);

    if ((user == NULL) || (pass == NULL)) {
        return (LM_AUTH_BAD_DATA);
    }

    if (type != SMTP_AUTH_LOGIN) {
        /* We only support login auth right now */
        return (LM_AUTH_UNSUPPORTED);
    }

    if ((retval = checkpassword(user, pass)) == 0) {
        return (LM_AUTH_OK);
    } else if (retval == -1) {
        return (LM_AUTH_ERROR);
    }
    return (LM_AUTH_FAILED);
}
